#!python.exe
# Much of this file borrowed from conda/core/portability.py:
#
# https://github.com/conda/conda/blob/master/conda/core/portability.py
#
# The license of which has been provided below:
#
# -----------------------------------------------------------------------------
#
# BSD 3-Clause License
#
# Copyright (c) 2012, Continuum Analytics, Inc.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#     * Redistributions of source code must retain the above copyright
#       notice, this list of conditions and the following disclaimer.
#     * Redistributions in binary form must reproduce the above copyright
#       notice, this list of conditions and the following disclaimer in the
#       documentation and/or other materials provided with the distribution.
#     * Neither the name of the copyright holder nor the names of its
#       contributors may be used to endorse or promote products
#       derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.

import platform
import re
import struct
import subprocess
import sys

on_win = sys.platform == 'win32'

# three capture groups: whole_shebang, executable, options
SHEBANG_REGEX = (
        # pretty much the whole match string
        br'^(#!'
        # allow spaces between #! and beginning of the executable path
        br'(?:[ ]*)'
        # the executable is the next text block without an escaped
        # space or non-space whitespace character
        br'(/(?:\\ |[^ \n\r\t])*)'
        # the rest of the line can contain option flags
        br'(.*)'
        # end whole_shebang group
        br')$')


def update_prefix(path, new_prefix, placeholder, mode='text'):
    if on_win and mode == 'text':
        # force all prefix replacements to forward slashes to simplify need to
        # escape backslashes replace with unix-style path separators
        new_prefix = new_prefix.replace('\\', '/')

    file_changed = False
    with open(path, 'rb+') as fh:
        original_data = fh.read()
        fh.seek(0)

        data = replace_prefix(original_data, mode, placeholder, new_prefix)

        # If the before and after content is the same, skip writing
        if data != original_data:
            fh.write(data)
            fh.truncate()
            file_changed = True

    if file_changed and platform.system() == "Darwin" and platform.machine() == "arm64":
        subprocess.run(
            ["/usr/bin/codesign", "-s", "-", "-f", path], capture_output=True
        )


def replace_prefix(data, mode, placeholder, new_prefix):
    if mode == 'text':
        data2 = text_replace(data, placeholder, new_prefix)
    elif mode == 'binary':
        data2 = binary_replace(data,
                               placeholder.encode('utf-8'),
                               new_prefix.encode('utf-8'))
        if not on_win and len(data2) != len(data):
            message = ("Found mismatched data length in binary file:\n"
                       "original data length: {len_orig!d})\n"
                       "new data length: {len_new!d}\n"
                       ).format(len_orig=len(data),
                                len_new=len(data2))
            raise ValueError(message)
    else:
        raise ValueError("Invalid mode: %r" % mode)
    return data2


def text_replace(data, placeholder, new_prefix):
    return data.replace(placeholder.encode('utf-8'), new_prefix.encode('utf-8'))


if on_win:
    def binary_replace(data, placeholder, new_prefix):
        new_prefix = new_prefix.lower()
        if placeholder in data:
            return replace_pyzzer_entry_point_shebang(data, placeholder, new_prefix)
        elif placeholder.lower() in data:
            return replace_pyzzer_entry_point_shebang(data, placeholder.lower(), new_prefix)
        return data

else:
    def binary_replace(data, placeholder, new_prefix):
        """Perform a binary replacement of `data`, where ``placeholder`` is
        replaced with ``new_prefix`` and the remaining string is padded with null
        characters.  All input arguments are expected to be bytes objects."""

        def replace(match):
            occurances = match.group().count(placeholder)
            padding = (len(placeholder) - len(new_prefix)) * occurances
            if padding < 0:
                raise ValueError("negative padding")
            return match.group().replace(placeholder, new_prefix) + b'\0' * padding

        pat = re.compile(re.escape(placeholder) + b'([^\0]*?)\0')
        return pat.sub(replace, data)


def replace_pyzzer_entry_point_shebang(all_data, placeholder, new_prefix):
    """Code adapted from pyzzer. This is meant to deal with entry point exe's
    created by distlib, which consist of a launcher, then a shebang, then a zip
    archive of the entry point code to run.  We need to change the shebang.
    """
    # Copyright (c) 2013 Vinay Sajip.
    #
    # Permission is hereby granted, free of charge, to any person obtaining a copy
    # of this software and associated documentation files (the "Software"), to deal
    # in the Software without restriction, including without limitation the rights
    # to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    # copies of the Software, and to permit persons to whom the Software is
    # furnished to do so, subject to the following conditions:
    #
    # The above copyright notice and this permission notice shall be included in
    # all copies or substantial portions of the Software.
    #
    # THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    # IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    # FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    # AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    # LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    # OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
    # THE SOFTWARE.
    launcher = shebang = None
    pos = all_data.rfind(b'PK\x05\x06')
    if pos >= 0:
        end_cdr = all_data[pos + 12:pos + 20]
        cdr_size, cdr_offset = struct.unpack('<LL', end_cdr)
        arc_pos = pos - cdr_size - cdr_offset
        data = all_data[arc_pos:]
        if arc_pos > 0:
            pos = all_data.rfind(b'#!', 0, arc_pos)
            if pos >= 0:
                shebang = all_data[pos:arc_pos]
                if pos > 0:
                    launcher = all_data[:pos]

        if data and shebang and launcher:
            if hasattr(placeholder, 'encode'):
                placeholder = placeholder.encode('utf-8')
            if hasattr(new_prefix, 'encode'):
                new_prefix = new_prefix.encode('utf-8')
            shebang = shebang.replace(placeholder, new_prefix)
            all_data = b"".join([launcher, shebang, data])
    return all_data


_prefix_records = [
('DLLs/_tkinter.pyd', 'C:\\b\\abs_e7pk5er3t2\\croot\\python-split_1733933824115\\_h_env', 'binary'),
('Lib\\site-packages\\__pycache__\\py.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\__pycache__\\typing_extensions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\_argcomplete.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\_version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\cacheprovider.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\capture.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\debugging.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\deprecated.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\doctest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\faulthandler.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\fixtures.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\freeze_support.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\helpconfig.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\hookspec.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\junitxml.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\legacypath.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\logging.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\main.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\monkeypatch.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\nodes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\outcomes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\pastebin.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\pathlib.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\pytester.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\pytester_assertions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\python.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\python_api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\python_path.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\recwarn.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\reports.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\runner.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\scope.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\setuponly.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\setupplan.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\skipping.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\stash.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\stepwise.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\terminal.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\threadexception.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\timing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\tmpdir.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\unittest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\unraisableexception.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\warning_types.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\__pycache__\\warnings.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\_argcomplete.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\_code\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\_code\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\_code\\__pycache__\\code.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\_code\\__pycache__\\source.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\_code\\code.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\_code\\source.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\_io\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\_io\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\_io\\__pycache__\\pprint.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\_io\\__pycache__\\saferepr.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\_io\\__pycache__\\terminalwriter.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\_io\\__pycache__\\wcwidth.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\_io\\pprint.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\_io\\saferepr.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\_io\\terminalwriter.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\_io\\wcwidth.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\_py\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\_py\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\_py\\__pycache__\\error.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\_py\\__pycache__\\path.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\_py\\error.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\_py\\path.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\_version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\assertion\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\assertion\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\assertion\\__pycache__\\rewrite.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\assertion\\__pycache__\\truncate.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\assertion\\__pycache__\\util.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\assertion\\rewrite.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\assertion\\truncate.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\assertion\\util.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\cacheprovider.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\capture.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\config\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\config\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\config\\__pycache__\\argparsing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\config\\__pycache__\\compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\config\\__pycache__\\exceptions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\config\\__pycache__\\findpaths.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\config\\argparsing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\config\\compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\config\\exceptions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\config\\findpaths.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\debugging.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\deprecated.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\doctest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\faulthandler.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\fixtures.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\freeze_support.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\helpconfig.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\hookspec.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\junitxml.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\legacypath.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\logging.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\main.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\mark\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\mark\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\mark\\__pycache__\\expression.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\mark\\__pycache__\\structures.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\_pytest\\mark\\expression.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\mark\\structures.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\monkeypatch.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\nodes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\outcomes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\pastebin.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\pathlib.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\pytester.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\pytester_assertions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\python.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\python_api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\python_path.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\recwarn.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\reports.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\runner.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\scope.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\setuponly.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\setupplan.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\skipping.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\stash.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\stepwise.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\terminal.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\threadexception.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\timing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\tmpdir.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\unittest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\unraisableexception.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\warning_types.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\_pytest\\warnings.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\annotated_types-0.7.0.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\annotated_types-0.7.0.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\annotated_types-0.7.0.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\annotated_types-0.7.0.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\annotated_types-0.7.0.dist-info\\licenses\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\annotated_types\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\annotated_types\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\annotated_types\\__pycache__\\test_cases.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\annotated_types\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\annotated_types\\test_cases.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio-4.8.0.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio-4.8.0.dist-info\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio-4.8.0.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio-4.8.0.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio-4.8.0.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio-4.8.0.dist-info\\entry_points.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio-4.8.0.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\__pycache__\\from_thread.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\__pycache__\\lowlevel.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\__pycache__\\pytest_plugin.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\__pycache__\\to_interpreter.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\__pycache__\\to_process.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\__pycache__\\to_thread.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\_backends\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\_backends\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\_backends\\__pycache__\\_asyncio.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\_backends\\__pycache__\\_trio.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\_backends\\_asyncio.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\_backends\\_trio.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\_core\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\_core\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\_core\\__pycache__\\_asyncio_selector_thread.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\_core\\__pycache__\\_eventloop.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\_core\\__pycache__\\_exceptions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\_core\\__pycache__\\_fileio.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\_core\\__pycache__\\_resources.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\_core\\__pycache__\\_signals.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\_core\\__pycache__\\_sockets.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\_core\\__pycache__\\_streams.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\_core\\__pycache__\\_subprocesses.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\_core\\__pycache__\\_synchronization.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\_core\\__pycache__\\_tasks.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\_core\\__pycache__\\_testing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\_core\\__pycache__\\_typedattr.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\_core\\_asyncio_selector_thread.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\_core\\_eventloop.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\_core\\_exceptions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\_core\\_fileio.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\_core\\_resources.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\_core\\_signals.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\_core\\_sockets.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\_core\\_streams.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\_core\\_subprocesses.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\_core\\_synchronization.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\_core\\_tasks.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\_core\\_testing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\_core\\_typedattr.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\abc\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\abc\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\abc\\__pycache__\\_eventloop.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\abc\\__pycache__\\_resources.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\abc\\__pycache__\\_sockets.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\abc\\__pycache__\\_streams.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\abc\\__pycache__\\_subprocesses.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\abc\\__pycache__\\_tasks.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\abc\\__pycache__\\_testing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\abc\\_eventloop.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\abc\\_resources.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\abc\\_sockets.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\abc\\_streams.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\abc\\_subprocesses.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\abc\\_tasks.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\abc\\_testing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\from_thread.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\lowlevel.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\pytest_plugin.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\streams\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\streams\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\streams\\__pycache__\\buffered.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\streams\\__pycache__\\file.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\streams\\__pycache__\\memory.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\streams\\__pycache__\\stapled.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\streams\\__pycache__\\text.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\streams\\__pycache__\\tls.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\anyio\\streams\\buffered.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\streams\\file.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\streams\\memory.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\streams\\stapled.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\streams\\text.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\streams\\tls.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\to_interpreter.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\to_process.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\anyio\\to_thread.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\click-8.1.8.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\click-8.1.8.dist-info\\LICENSE.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\click-8.1.8.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\click-8.1.8.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\click-8.1.8.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\click\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\click\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\click\\__pycache__\\_compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\click\\__pycache__\\_termui_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\click\\__pycache__\\_textwrap.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\click\\__pycache__\\_winconsole.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\click\\__pycache__\\core.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\click\\__pycache__\\decorators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\click\\__pycache__\\exceptions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\click\\__pycache__\\formatting.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\click\\__pycache__\\globals.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\click\\__pycache__\\parser.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\click\\__pycache__\\shell_completion.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\click\\__pycache__\\termui.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\click\\__pycache__\\testing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\click\\__pycache__\\types.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\click\\__pycache__\\utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\click\\_compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\click\\_termui_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\click\\_textwrap.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\click\\_winconsole.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\click\\core.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\click\\decorators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\click\\exceptions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\click\\formatting.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\click\\globals.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\click\\parser.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\click\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\click\\shell_completion.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\click\\termui.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\click\\testing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\click\\types.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\click\\utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\colorama-0.4.6.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\colorama-0.4.6.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\colorama-0.4.6.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\colorama-0.4.6.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\colorama-0.4.6.dist-info\\licenses\\LICENSE.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\colorama\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\colorama\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\colorama\\__pycache__\\ansi.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\colorama\\__pycache__\\ansitowin32.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\colorama\\__pycache__\\initialise.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\colorama\\__pycache__\\win32.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\colorama\\__pycache__\\winterm.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\colorama\\ansi.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\colorama\\ansitowin32.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\colorama\\initialise.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\colorama\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\colorama\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\colorama\\tests\\__pycache__\\ansi_test.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\colorama\\tests\\__pycache__\\ansitowin32_test.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\colorama\\tests\\__pycache__\\initialise_test.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\colorama\\tests\\__pycache__\\isatty_test.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\colorama\\tests\\__pycache__\\utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\colorama\\tests\\__pycache__\\winterm_test.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\colorama\\tests\\ansi_test.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\colorama\\tests\\ansitowin32_test.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\colorama\\tests\\initialise_test.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\colorama\\tests\\isatty_test.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\colorama\\tests\\utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\colorama\\tests\\winterm_test.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\colorama\\win32.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\colorama\\winterm.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\dotenv\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\dotenv\\__main__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\dotenv\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\dotenv\\__pycache__\\__main__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\dotenv\\__pycache__\\cli.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\dotenv\\__pycache__\\ipython.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\dotenv\\__pycache__\\main.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\dotenv\\__pycache__\\parser.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\dotenv\\__pycache__\\variables.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\dotenv\\__pycache__\\version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\dotenv\\cli.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\dotenv\\ipython.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\dotenv\\main.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\dotenv\\parser.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\dotenv\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\dotenv\\variables.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\dotenv\\version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\exceptiongroup-1.2.2.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\exceptiongroup-1.2.2.dist-info\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\exceptiongroup-1.2.2.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\exceptiongroup-1.2.2.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\exceptiongroup-1.2.2.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\exceptiongroup\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\exceptiongroup\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\exceptiongroup\\__pycache__\\_catch.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\exceptiongroup\\__pycache__\\_exceptions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\exceptiongroup\\__pycache__\\_formatting.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\exceptiongroup\\__pycache__\\_suppress.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\exceptiongroup\\__pycache__\\_version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\exceptiongroup\\_catch.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\exceptiongroup\\_exceptions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\exceptiongroup\\_formatting.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\exceptiongroup\\_suppress.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\exceptiongroup\\_version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\exceptiongroup\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi-0.115.11.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi-0.115.11.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi-0.115.11.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi-0.115.11.dist-info\\REQUESTED', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi-0.115.11.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi-0.115.11.dist-info\\entry_points.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi-0.115.11.dist-info\\licenses\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\__main__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\__pycache__\\__main__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\__pycache__\\_compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\__pycache__\\applications.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\__pycache__\\background.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\__pycache__\\cli.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\__pycache__\\concurrency.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\__pycache__\\datastructures.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\__pycache__\\encoders.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\__pycache__\\exception_handlers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\__pycache__\\exceptions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\__pycache__\\logger.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\__pycache__\\param_functions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\__pycache__\\params.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\__pycache__\\requests.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\__pycache__\\responses.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\__pycache__\\routing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\__pycache__\\staticfiles.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\__pycache__\\templating.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\__pycache__\\testclient.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\__pycache__\\types.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\__pycache__\\utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\__pycache__\\websockets.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\_compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\applications.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\background.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\cli.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\concurrency.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\datastructures.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\dependencies\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\dependencies\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\dependencies\\__pycache__\\models.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\dependencies\\__pycache__\\utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\dependencies\\models.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\dependencies\\utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\encoders.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\exception_handlers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\exceptions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\logger.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\middleware\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\middleware\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\middleware\\__pycache__\\cors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\middleware\\__pycache__\\gzip.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\middleware\\__pycache__\\httpsredirect.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\middleware\\__pycache__\\trustedhost.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\middleware\\__pycache__\\wsgi.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\middleware\\cors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\middleware\\gzip.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\middleware\\httpsredirect.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\middleware\\trustedhost.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\middleware\\wsgi.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\openapi\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\openapi\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\openapi\\__pycache__\\constants.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\openapi\\__pycache__\\docs.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\openapi\\__pycache__\\models.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\openapi\\__pycache__\\utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\openapi\\constants.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\openapi\\docs.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\openapi\\models.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\openapi\\utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\param_functions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\params.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\requests.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\responses.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\routing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\security\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\security\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\security\\__pycache__\\api_key.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\security\\__pycache__\\base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\security\\__pycache__\\http.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\security\\__pycache__\\oauth2.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\security\\__pycache__\\open_id_connect_url.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\security\\__pycache__\\utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\fastapi\\security\\api_key.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\security\\base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\security\\http.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\security\\oauth2.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\security\\open_id_connect_url.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\security\\utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\staticfiles.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\templating.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\testclient.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\types.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\fastapi\\websockets.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11-0.14.0.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11-0.14.0.dist-info\\LICENSE.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11-0.14.0.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11-0.14.0.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11-0.14.0.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11-0.14.0.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\h11\\__pycache__\\_abnf.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\h11\\__pycache__\\_connection.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\h11\\__pycache__\\_events.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\h11\\__pycache__\\_headers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\h11\\__pycache__\\_readers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\h11\\__pycache__\\_receivebuffer.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\h11\\__pycache__\\_state.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\h11\\__pycache__\\_util.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\h11\\__pycache__\\_version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\h11\\__pycache__\\_writers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\h11\\_abnf.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\_connection.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\_events.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\_headers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\_readers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\_receivebuffer.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\_state.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\_util.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\_version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\_writers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\h11\\tests\\__pycache__\\helpers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\h11\\tests\\__pycache__\\test_against_stdlib_http.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\h11\\tests\\__pycache__\\test_connection.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\h11\\tests\\__pycache__\\test_events.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\h11\\tests\\__pycache__\\test_headers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\h11\\tests\\__pycache__\\test_helpers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\h11\\tests\\__pycache__\\test_io.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\h11\\tests\\__pycache__\\test_receivebuffer.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\h11\\tests\\__pycache__\\test_state.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\h11\\tests\\__pycache__\\test_util.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\h11\\tests\\data\\test-file', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\tests\\helpers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\tests\\test_against_stdlib_http.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\tests\\test_connection.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\tests\\test_events.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\tests\\test_headers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\tests\\test_helpers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\tests\\test_io.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\tests\\test_receivebuffer.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\tests\\test_state.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\h11\\tests\\test_util.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\idna-3.10.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\idna-3.10.dist-info\\LICENSE.md', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\idna-3.10.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\idna-3.10.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\idna-3.10.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\idna\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\idna\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\idna\\__pycache__\\codec.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\idna\\__pycache__\\compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\idna\\__pycache__\\core.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\idna\\__pycache__\\idnadata.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\idna\\__pycache__\\intranges.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\idna\\__pycache__\\package_data.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\idna\\__pycache__\\uts46data.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\idna\\codec.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\idna\\compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\idna\\core.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\idna\\idnadata.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\idna\\intranges.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\idna\\package_data.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\idna\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\idna\\uts46data.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\iniconfig-2.0.0.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\iniconfig-2.0.0.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\iniconfig-2.0.0.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\iniconfig-2.0.0.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\iniconfig-2.0.0.dist-info\\licenses\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\iniconfig\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\iniconfig\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\iniconfig\\__pycache__\\_parse.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\iniconfig\\__pycache__\\_version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\iniconfig\\__pycache__\\exceptions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\iniconfig\\_parse.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\iniconfig\\_version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\iniconfig\\exceptions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\iniconfig\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\multipart\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\multipart\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\multipart\\__pycache__\\decoders.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\multipart\\__pycache__\\exceptions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\multipart\\__pycache__\\multipart.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\multipart\\decoders.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\multipart\\exceptions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\multipart\\multipart.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging-24.2.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging-24.2.dist-info\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging-24.2.dist-info\\LICENSE.APACHE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging-24.2.dist-info\\LICENSE.BSD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging-24.2.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging-24.2.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging-24.2.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\_elffile.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\_manylinux.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\_musllinux.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\_parser.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\_structures.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\_tokenizer.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\markers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\metadata.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\requirements.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\specifiers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\tags.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\packaging\\_elffile.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging\\_manylinux.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging\\_musllinux.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging\\_parser.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging\\_structures.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging\\_tokenizer.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging\\licenses\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging\\licenses\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\packaging\\licenses\\__pycache__\\_spdx.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\packaging\\licenses\\_spdx.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging\\markers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging\\metadata.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging\\requirements.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging\\specifiers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging\\tags.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging\\utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\packaging\\version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pluggy-1.5.0.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pluggy-1.5.0.dist-info\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pluggy-1.5.0.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pluggy-1.5.0.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pluggy-1.5.0.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pluggy-1.5.0.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pluggy\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pluggy\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pluggy\\__pycache__\\_callers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pluggy\\__pycache__\\_hooks.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pluggy\\__pycache__\\_manager.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pluggy\\__pycache__\\_result.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pluggy\\__pycache__\\_tracing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pluggy\\__pycache__\\_version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pluggy\\__pycache__\\_warnings.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pluggy\\_callers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pluggy\\_hooks.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pluggy\\_manager.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pluggy\\_result.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pluggy\\_tracing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pluggy\\_version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pluggy\\_warnings.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pluggy\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\py.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic-2.10.6.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic-2.10.6.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic-2.10.6.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic-2.10.6.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic-2.10.6.dist-info\\licenses\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\_migration.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\alias_generators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\aliases.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\annotated_handlers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\class_validators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\color.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\config.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\dataclasses.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\datetime_parse.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\decorator.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\env_settings.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\error_wrappers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\errors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\fields.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\functional_serializers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\functional_validators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\generics.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\json.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\json_schema.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\main.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\mypy.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\networks.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\parse.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\root_model.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\schema.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\tools.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\type_adapter.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\types.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\typing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\validate_call_decorator.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\validators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\warnings.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_config.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_core_metadata.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_core_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_dataclasses.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_decorators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_decorators_v1.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_discriminated_union.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_docs_extraction.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_fields.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_forward_ref.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_generate_schema.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_generics.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_git.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_import_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_internal_dataclass.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_known_annotated_metadata.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_mock_val_ser.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_model_construction.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_namespace_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_repr.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_schema_generation_shared.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_serializers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_signature.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_std_types_schema.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_typing_extra.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_validate_call.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_validators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\_config.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_core_metadata.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_core_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_dataclasses.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_decorators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_decorators_v1.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_discriminated_union.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_docs_extraction.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_fields.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_forward_ref.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_generate_schema.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_generics.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_git.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_import_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_internal_dataclass.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_known_annotated_metadata.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_mock_val_ser.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_model_construction.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_namespace_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_repr.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_schema_generation_shared.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_serializers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_signature.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_std_types_schema.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_typing_extra.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_validate_call.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_validators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\_migration.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\alias_generators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\aliases.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\annotated_handlers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\class_validators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\color.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\config.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\dataclasses.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\datetime_parse.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\decorator.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\deprecated\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\deprecated\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\deprecated\\__pycache__\\class_validators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\deprecated\\__pycache__\\config.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\deprecated\\__pycache__\\copy_internals.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\deprecated\\__pycache__\\decorator.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\deprecated\\__pycache__\\json.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\deprecated\\__pycache__\\parse.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\deprecated\\__pycache__\\tools.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\deprecated\\class_validators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\deprecated\\config.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\deprecated\\copy_internals.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\deprecated\\decorator.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\deprecated\\json.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\deprecated\\parse.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\deprecated\\tools.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\env_settings.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\error_wrappers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\errors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\experimental\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\experimental\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\experimental\\__pycache__\\pipeline.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\experimental\\pipeline.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\fields.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\functional_serializers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\functional_validators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\generics.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\json.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\json_schema.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\main.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\mypy.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\networks.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\parse.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\plugin\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\plugin\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\plugin\\__pycache__\\_loader.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\plugin\\__pycache__\\_schema_validator.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\plugin\\_loader.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\plugin\\_schema_validator.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\root_model.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\schema.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\tools.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\type_adapter.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\types.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\typing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\_hypothesis_plugin.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\annotated_types.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\class_validators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\color.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\config.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\dataclasses.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\datetime_parse.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\decorator.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\env_settings.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\error_wrappers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\errors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\fields.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\generics.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\json.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\main.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\mypy.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\networks.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\parse.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\schema.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\tools.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\types.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\typing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\validators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\_hypothesis_plugin.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\annotated_types.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\class_validators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\color.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\config.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\dataclasses.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\datetime_parse.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\decorator.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\env_settings.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\error_wrappers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\errors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\fields.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\generics.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\json.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\main.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\mypy.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\networks.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\parse.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\schema.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\tools.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\types.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\typing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\validators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\v1\\version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\validate_call_decorator.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\validators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic\\warnings.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic_core-2.27.2.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic_core-2.27.2.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic_core-2.27.2.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic_core-2.27.2.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic_core-2.27.2.dist-info\\licenses\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic_core\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic_core\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic_core\\__pycache__\\core_schema.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic_core\\_pydantic_core.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pydantic_core\\_pydantic_core.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic_core\\core_schema.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pydantic_core\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pytest-8.3.5.dist-info\\AUTHORS', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pytest-8.3.5.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pytest-8.3.5.dist-info\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pytest-8.3.5.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pytest-8.3.5.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pytest-8.3.5.dist-info\\REQUESTED', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pytest-8.3.5.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pytest-8.3.5.dist-info\\entry_points.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pytest-8.3.5.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pytest\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pytest\\__main__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\pytest\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pytest\\__pycache__\\__main__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\pytest\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\python_dotenv-1.0.1.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\python_dotenv-1.0.1.dist-info\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\python_dotenv-1.0.1.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\python_dotenv-1.0.1.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\python_dotenv-1.0.1.dist-info\\REQUESTED', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\python_dotenv-1.0.1.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\python_dotenv-1.0.1.dist-info\\entry_points.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\python_dotenv-1.0.1.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\python_multipart-0.0.20.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\python_multipart-0.0.20.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\python_multipart-0.0.20.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\python_multipart-0.0.20.dist-info\\REQUESTED', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\python_multipart-0.0.20.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\python_multipart-0.0.20.dist-info\\licenses\\LICENSE.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\python_multipart\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\python_multipart\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\python_multipart\\__pycache__\\decoders.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\python_multipart\\__pycache__\\exceptions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\python_multipart\\__pycache__\\multipart.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\python_multipart\\decoders.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\python_multipart\\exceptions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\python_multipart\\multipart.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\python_multipart\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\sniffio-1.3.1.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\sniffio-1.3.1.dist-info\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\sniffio-1.3.1.dist-info\\LICENSE.APACHE2', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\sniffio-1.3.1.dist-info\\LICENSE.MIT', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\sniffio-1.3.1.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\sniffio-1.3.1.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\sniffio-1.3.1.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\sniffio-1.3.1.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\sniffio\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\sniffio\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\sniffio\\__pycache__\\_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\sniffio\\__pycache__\\_version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\sniffio\\_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\sniffio\\_tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\sniffio\\_tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\sniffio\\_tests\\__pycache__\\test_sniffio.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\sniffio\\_tests\\test_sniffio.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\sniffio\\_version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\sniffio\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette-0.46.0.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette-0.46.0.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette-0.46.0.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette-0.46.0.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette-0.46.0.dist-info\\licenses\\LICENSE.md', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\__pycache__\\_exception_handler.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\__pycache__\\_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\__pycache__\\applications.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\__pycache__\\authentication.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\__pycache__\\background.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\__pycache__\\concurrency.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\__pycache__\\config.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\__pycache__\\convertors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\__pycache__\\datastructures.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\__pycache__\\endpoints.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\__pycache__\\exceptions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\__pycache__\\formparsers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\__pycache__\\requests.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\__pycache__\\responses.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\__pycache__\\routing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\__pycache__\\schemas.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\__pycache__\\staticfiles.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\__pycache__\\status.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\__pycache__\\templating.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\__pycache__\\testclient.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\__pycache__\\types.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\__pycache__\\websockets.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\_exception_handler.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\applications.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\authentication.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\background.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\concurrency.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\config.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\convertors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\datastructures.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\endpoints.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\exceptions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\formparsers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\middleware\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\middleware\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\middleware\\__pycache__\\authentication.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\middleware\\__pycache__\\base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\middleware\\__pycache__\\cors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\middleware\\__pycache__\\errors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\middleware\\__pycache__\\exceptions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\middleware\\__pycache__\\gzip.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\middleware\\__pycache__\\httpsredirect.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\middleware\\__pycache__\\sessions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\middleware\\__pycache__\\trustedhost.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\middleware\\__pycache__\\wsgi.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\starlette\\middleware\\authentication.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\middleware\\base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\middleware\\cors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\middleware\\errors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\middleware\\exceptions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\middleware\\gzip.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\middleware\\httpsredirect.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\middleware\\sessions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\middleware\\trustedhost.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\middleware\\wsgi.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\requests.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\responses.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\routing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\schemas.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\staticfiles.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\status.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\templating.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\testclient.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\types.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\starlette\\websockets.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\tomli-2.2.1.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\tomli-2.2.1.dist-info\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\tomli-2.2.1.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\tomli-2.2.1.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\tomli-2.2.1.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\tomli\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\tomli\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\tomli\\__pycache__\\_parser.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\tomli\\__pycache__\\_re.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\tomli\\__pycache__\\_types.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\tomli\\_parser.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\tomli\\_re.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\tomli\\_types.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\tomli\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\typing_extensions-4.12.2.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\typing_extensions-4.12.2.dist-info\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\typing_extensions-4.12.2.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\typing_extensions-4.12.2.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\typing_extensions-4.12.2.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\typing_extensions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn-0.34.0.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn-0.34.0.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn-0.34.0.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn-0.34.0.dist-info\\REQUESTED', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn-0.34.0.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn-0.34.0.dist-info\\entry_points.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn-0.34.0.dist-info\\licenses\\LICENSE.md', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\__main__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\__pycache__\\__main__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\__pycache__\\_subprocess.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\__pycache__\\_types.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\__pycache__\\config.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\__pycache__\\importer.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\__pycache__\\logging.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\__pycache__\\main.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\__pycache__\\server.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\__pycache__\\workers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\_subprocess.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\_types.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\config.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\importer.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\lifespan\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\lifespan\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\lifespan\\__pycache__\\off.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\lifespan\\__pycache__\\on.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\lifespan\\off.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\lifespan\\on.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\logging.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\loops\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\loops\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\loops\\__pycache__\\asyncio.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\loops\\__pycache__\\auto.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\loops\\__pycache__\\uvloop.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\loops\\asyncio.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\loops\\auto.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\loops\\uvloop.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\main.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\middleware\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\middleware\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\middleware\\__pycache__\\asgi2.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\middleware\\__pycache__\\message_logger.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\middleware\\__pycache__\\proxy_headers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\middleware\\__pycache__\\wsgi.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\middleware\\asgi2.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\middleware\\message_logger.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\middleware\\proxy_headers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\middleware\\wsgi.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\protocols\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\protocols\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\protocols\\__pycache__\\utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\protocols\\http\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\protocols\\http\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\protocols\\http\\__pycache__\\auto.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\protocols\\http\\__pycache__\\flow_control.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\protocols\\http\\__pycache__\\h11_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\protocols\\http\\__pycache__\\httptools_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\protocols\\http\\auto.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\protocols\\http\\flow_control.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\protocols\\http\\h11_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\protocols\\http\\httptools_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\protocols\\utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\protocols\\websockets\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\protocols\\websockets\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\protocols\\websockets\\__pycache__\\auto.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\protocols\\websockets\\__pycache__\\websockets_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\protocols\\websockets\\__pycache__\\wsproto_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\protocols\\websockets\\auto.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\protocols\\websockets\\websockets_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\protocols\\websockets\\wsproto_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\server.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\supervisors\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\supervisors\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\supervisors\\__pycache__\\basereload.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\supervisors\\__pycache__\\multiprocess.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\supervisors\\__pycache__\\statreload.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\supervisors\\__pycache__\\watchfilesreload.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Lib\\site-packages\\uvicorn\\supervisors\\basereload.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\supervisors\\multiprocess.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\supervisors\\statreload.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\supervisors\\watchfilesreload.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Lib\\site-packages\\uvicorn\\workers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Library/bin/c_rehash.pl', 'C:\\\\b\\\\abs_04f5aioy9i\\\\croot\\\\openssl_1740989503961\\\\_h_env', 'text'),
('Library/bin/libcrypto-3-x64.dll', 'C:\\b\\abs_04f5aioy9i\\croot\\openssl_1740989503961\\_h_env', 'binary'),
('Library/bin/libcrypto-3-x64.pdb', 'C:\\\\b\\\\abs_04f5aioy9i\\\\croot\\\\openssl_1740989503961\\\\_h_env', 'binary'),
('Library/bin/libssl-3-x64.pdb', 'C:\\\\b\\\\abs_04f5aioy9i\\\\croot\\\\openssl_1740989503961\\\\_h_env', 'binary'),
('Library/bin/openssl.pdb', 'C:\\\\b\\\\abs_04f5aioy9i\\\\croot\\\\openssl_1740989503961\\\\_h_env', 'binary'),
('Library/bin/tcl86t.dll', 'C:\\b\\abs_17sr80_iza\\croot\\tk_1714770602079\\_h_env', 'binary'),
('Library/lib/pkgconfig/libcrypto.pc', 'C:/b/abs_04f5aioy9i/croot/openssl_1740989503961/_h_env', 'text'),
('Library/lib/pkgconfig/liblzma.pc', 'C:/b/abs_127uq9r7f9/croot/xz_1739468293592/_h_env', 'text'),
('Library/lib/pkgconfig/libssl.pc', 'C:/b/abs_04f5aioy9i/croot/openssl_1740989503961/_h_env', 'text'),
('Library/lib/pkgconfig/openssl.pc', 'C:/b/abs_04f5aioy9i/croot/openssl_1740989503961/_h_env', 'text'),
('Library/lib/pkgconfig/zlib.pc', 'C:/b/abs_72khlqt_kh/croot/zlib_1714514696139/_h_env', 'text'),
('Library/lib/tclConfig.sh', 'C:\\b\\abs_17sr80_iza\\croot\\tk_1714770602079\\_h_env', 'text'),
('Scripts/conda-pack-script.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'text'),
('Scripts/conda-pack.exe', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Scripts\\dotenv.exe', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Scripts\\fastapi.exe', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Scripts\\py.test.exe', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Scripts\\pytest.exe', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('Scripts\\uvicorn.exe', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\mock_tcp', 'binary'),
('python.pdb', 'C:\\b\\abs_e7pk5er3t2\\croot\\python-split_1733933824115\\_h_env', 'binary'),
('python310.pdb', 'C:\\b\\abs_e7pk5er3t2\\croot\\python-split_1733933824115\\_h_env', 'binary'),
('pythonw.pdb', 'C:\\b\\abs_e7pk5er3t2\\croot\\python-split_1733933824115\\_h_env', 'binary')
]

if __name__ == '__main__':
    import os
    import argparse
    parser = argparse.ArgumentParser(
            prog='conda-unpack',
            description=('Finish unpacking the environment after unarchiving. '
                         'Cleans up absolute prefixes in any remaining files'))
    parser.add_argument('--version',
                        action='store_true',
                        help='Show version then exit')
    args = parser.parse_args()
    # Manually handle version printing to output to stdout in python < 3.4
    if args.version:
        print('conda-unpack 0.8.1')
    else:
        script_dir = os.path.dirname(__file__)
        new_prefix = os.path.abspath(os.path.dirname(script_dir))
        for path, placeholder, mode in _prefix_records:
            new_path = os.path.join(new_prefix, path)
            if on_win:
                new_path = new_path.replace('\\', '/')
            update_prefix(new_path, new_prefix, placeholder, mode=mode)
