#!python.exe
# Much of this file borrowed from conda/core/portability.py:
#
# https://github.com/conda/conda/blob/master/conda/core/portability.py
#
# The license of which has been provided below:
#
# -----------------------------------------------------------------------------
#
# BSD 3-Clause License
#
# Copyright (c) 2012, Continuum Analytics, Inc.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#     * Redistributions of source code must retain the above copyright
#       notice, this list of conditions and the following disclaimer.
#     * Redistributions in binary form must reproduce the above copyright
#       notice, this list of conditions and the following disclaimer in the
#       documentation and/or other materials provided with the distribution.
#     * Neither the name of the copyright holder nor the names of its
#       contributors may be used to endorse or promote products
#       derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.

import platform
import re
import struct
import subprocess
import sys

on_win = sys.platform == 'win32'

# three capture groups: whole_shebang, executable, options
SHEBANG_REGEX = (
        # pretty much the whole match string
        br'^(#!'
        # allow spaces between #! and beginning of the executable path
        br'(?:[ ]*)'
        # the executable is the next text block without an escaped
        # space or non-space whitespace character
        br'(/(?:\\ |[^ \n\r\t])*)'
        # the rest of the line can contain option flags
        br'(.*)'
        # end whole_shebang group
        br')$')


def update_prefix(path, new_prefix, placeholder, mode='text'):
    if on_win and mode == 'text':
        # force all prefix replacements to forward slashes to simplify need to
        # escape backslashes replace with unix-style path separators
        new_prefix = new_prefix.replace('\\', '/')

    file_changed = False
    with open(path, 'rb+') as fh:
        original_data = fh.read()
        fh.seek(0)

        data = replace_prefix(original_data, mode, placeholder, new_prefix)

        # If the before and after content is the same, skip writing
        if data != original_data:
            fh.write(data)
            fh.truncate()
            file_changed = True

    if file_changed and platform.system() == "Darwin" and platform.machine() == "arm64":
        subprocess.run(
            ["/usr/bin/codesign", "-s", "-", "-f", path], capture_output=True
        )


def replace_prefix(data, mode, placeholder, new_prefix):
    if mode == 'text':
        data2 = text_replace(data, placeholder, new_prefix)
    elif mode == 'binary':
        data2 = binary_replace(data,
                               placeholder.encode('utf-8'),
                               new_prefix.encode('utf-8'))
        if not on_win and len(data2) != len(data):
            message = ("Found mismatched data length in binary file:\n"
                       "original data length: {len_orig!d})\n"
                       "new data length: {len_new!d}\n"
                       ).format(len_orig=len(data),
                                len_new=len(data2))
            raise ValueError(message)
    else:
        raise ValueError("Invalid mode: %r" % mode)
    return data2


def text_replace(data, placeholder, new_prefix):
    return data.replace(placeholder.encode('utf-8'), new_prefix.encode('utf-8'))


if on_win:
    def binary_replace(data, placeholder, new_prefix):
        new_prefix = new_prefix.lower()
        if placeholder in data:
            return replace_pyzzer_entry_point_shebang(data, placeholder, new_prefix)
        elif placeholder.lower() in data:
            return replace_pyzzer_entry_point_shebang(data, placeholder.lower(), new_prefix)
        return data

else:
    def binary_replace(data, placeholder, new_prefix):
        """Perform a binary replacement of `data`, where ``placeholder`` is
        replaced with ``new_prefix`` and the remaining string is padded with null
        characters.  All input arguments are expected to be bytes objects."""

        def replace(match):
            occurances = match.group().count(placeholder)
            padding = (len(placeholder) - len(new_prefix)) * occurances
            if padding < 0:
                raise ValueError("negative padding")
            return match.group().replace(placeholder, new_prefix) + b'\0' * padding

        pat = re.compile(re.escape(placeholder) + b'([^\0]*?)\0')
        return pat.sub(replace, data)


def replace_pyzzer_entry_point_shebang(all_data, placeholder, new_prefix):
    """Code adapted from pyzzer. This is meant to deal with entry point exe's
    created by distlib, which consist of a launcher, then a shebang, then a zip
    archive of the entry point code to run.  We need to change the shebang.
    """
    # Copyright (c) 2013 Vinay Sajip.
    #
    # Permission is hereby granted, free of charge, to any person obtaining a copy
    # of this software and associated documentation files (the "Software"), to deal
    # in the Software without restriction, including without limitation the rights
    # to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    # copies of the Software, and to permit persons to whom the Software is
    # furnished to do so, subject to the following conditions:
    #
    # The above copyright notice and this permission notice shall be included in
    # all copies or substantial portions of the Software.
    #
    # THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    # IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    # FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    # AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    # LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    # OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
    # THE SOFTWARE.
    launcher = shebang = None
    pos = all_data.rfind(b'PK\x05\x06')
    if pos >= 0:
        end_cdr = all_data[pos + 12:pos + 20]
        cdr_size, cdr_offset = struct.unpack('<LL', end_cdr)
        arc_pos = pos - cdr_size - cdr_offset
        data = all_data[arc_pos:]
        if arc_pos > 0:
            pos = all_data.rfind(b'#!', 0, arc_pos)
            if pos >= 0:
                shebang = all_data[pos:arc_pos]
                if pos > 0:
                    launcher = all_data[:pos]

        if data and shebang and launcher:
            if hasattr(placeholder, 'encode'):
                placeholder = placeholder.encode('utf-8')
            if hasattr(new_prefix, 'encode'):
                new_prefix = new_prefix.encode('utf-8')
            shebang = shebang.replace(placeholder, new_prefix)
            all_data = b"".join([launcher, shebang, data])
    return all_data


_prefix_records = [
('DLLs/_tkinter.pyd', 'C:\\b\\abs_e7pk5er3t2\\croot\\python-split_1733933824115\\_h_env', 'binary'),
('Lib\\site-packages\\GeoAlchemy2-0.17.1.dist-info\\COPYING.rst', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\GeoAlchemy2-0.17.1.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\GeoAlchemy2-0.17.1.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\GeoAlchemy2-0.17.1.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\GeoAlchemy2-0.17.1.dist-info\\REQUESTED', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\GeoAlchemy2-0.17.1.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\GeoAlchemy2-0.17.1.dist-info\\entry_points.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\GeoAlchemy2-0.17.1.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\SQLAlchemy-2.0.38.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\SQLAlchemy-2.0.38.dist-info\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\SQLAlchemy-2.0.38.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\SQLAlchemy-2.0.38.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\SQLAlchemy-2.0.38.dist-info\\REQUESTED', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\SQLAlchemy-2.0.38.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\SQLAlchemy-2.0.38.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\__pycache__\\six.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\__pycache__\\typing_extensions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\annotated_types-0.7.0.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\annotated_types-0.7.0.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\annotated_types-0.7.0.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\annotated_types-0.7.0.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\annotated_types-0.7.0.dist-info\\licenses\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\annotated_types\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\annotated_types\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\annotated_types\\__pycache__\\test_cases.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\annotated_types\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\annotated_types\\test_cases.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\certifi-2025.1.31.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\certifi-2025.1.31.dist-info\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\certifi-2025.1.31.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\certifi-2025.1.31.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\certifi-2025.1.31.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\certifi-2025.1.31.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\certifi\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\certifi\\__main__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\certifi\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\certifi\\__pycache__\\__main__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\certifi\\__pycache__\\core.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\certifi\\cacert.pem', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\certifi\\core.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\certifi\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\charset_normalizer-3.4.1.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\charset_normalizer-3.4.1.dist-info\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\charset_normalizer-3.4.1.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\charset_normalizer-3.4.1.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\charset_normalizer-3.4.1.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\charset_normalizer-3.4.1.dist-info\\entry_points.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\charset_normalizer-3.4.1.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\charset_normalizer\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\charset_normalizer\\__main__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\charset_normalizer\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\charset_normalizer\\__pycache__\\__main__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\charset_normalizer\\__pycache__\\api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\charset_normalizer\\__pycache__\\cd.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\charset_normalizer\\__pycache__\\constant.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\charset_normalizer\\__pycache__\\legacy.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\charset_normalizer\\__pycache__\\md.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\charset_normalizer\\__pycache__\\models.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\charset_normalizer\\__pycache__\\utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\charset_normalizer\\__pycache__\\version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\charset_normalizer\\api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\charset_normalizer\\cd.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\charset_normalizer\\cli\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\charset_normalizer\\cli\\__main__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\charset_normalizer\\cli\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\charset_normalizer\\cli\\__pycache__\\__main__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\charset_normalizer\\constant.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\charset_normalizer\\legacy.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\charset_normalizer\\md.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\charset_normalizer\\md.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\charset_normalizer\\md__mypyc.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\charset_normalizer\\models.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\charset_normalizer\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\charset_normalizer\\utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\charset_normalizer\\version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dateutil\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dateutil\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dateutil\\__pycache__\\_common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dateutil\\__pycache__\\_version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dateutil\\__pycache__\\easter.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dateutil\\__pycache__\\relativedelta.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dateutil\\__pycache__\\rrule.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dateutil\\__pycache__\\tzwin.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dateutil\\__pycache__\\utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dateutil\\_common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dateutil\\_version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dateutil\\easter.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dateutil\\parser\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dateutil\\parser\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dateutil\\parser\\__pycache__\\_parser.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dateutil\\parser\\__pycache__\\isoparser.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dateutil\\parser\\_parser.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dateutil\\parser\\isoparser.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dateutil\\relativedelta.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dateutil\\rrule.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dateutil\\tz\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dateutil\\tz\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dateutil\\tz\\__pycache__\\_common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dateutil\\tz\\__pycache__\\_factories.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dateutil\\tz\\__pycache__\\tz.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dateutil\\tz\\__pycache__\\win.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dateutil\\tz\\_common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dateutil\\tz\\_factories.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dateutil\\tz\\tz.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dateutil\\tz\\win.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dateutil\\tzwin.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dateutil\\utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dateutil\\zoneinfo\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dateutil\\zoneinfo\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dateutil\\zoneinfo\\__pycache__\\rebuild.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dateutil\\zoneinfo\\dateutil-zoneinfo.tar.gz', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dateutil\\zoneinfo\\rebuild.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dotenv\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dotenv\\__main__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dotenv\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dotenv\\__pycache__\\__main__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dotenv\\__pycache__\\cli.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dotenv\\__pycache__\\ipython.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dotenv\\__pycache__\\main.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dotenv\\__pycache__\\parser.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dotenv\\__pycache__\\variables.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dotenv\\__pycache__\\version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\dotenv\\cli.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dotenv\\ipython.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dotenv\\main.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dotenv\\parser.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dotenv\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dotenv\\variables.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\dotenv\\version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future-1.0.0.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future-1.0.0.dist-info\\LICENSE.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future-1.0.0.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future-1.0.0.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future-1.0.0.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future-1.0.0.dist-info\\entry_points.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future-1.0.0.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\__pycache__\\_markupbase.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\__pycache__\\datetime.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\__pycache__\\misc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\__pycache__\\socket.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\__pycache__\\socketserver.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\__pycache__\\total_ordering.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\_markupbase.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\datetime.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\__pycache__\\_encoded_words.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\__pycache__\\_header_value_parser.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\__pycache__\\_parseaddr.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\__pycache__\\_policybase.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\__pycache__\\base64mime.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\__pycache__\\charset.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\__pycache__\\encoders.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\__pycache__\\errors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\__pycache__\\feedparser.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\__pycache__\\generator.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\__pycache__\\header.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\__pycache__\\headerregistry.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\__pycache__\\iterators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\__pycache__\\message.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\__pycache__\\parser.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\__pycache__\\policy.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\__pycache__\\quoprimime.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\__pycache__\\utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\_encoded_words.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\_header_value_parser.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\_parseaddr.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\_policybase.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\base64mime.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\charset.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\encoders.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\errors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\feedparser.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\generator.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\header.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\headerregistry.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\iterators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\message.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\mime\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\mime\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\mime\\__pycache__\\application.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\mime\\__pycache__\\audio.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\mime\\__pycache__\\base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\mime\\__pycache__\\image.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\mime\\__pycache__\\message.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\mime\\__pycache__\\multipart.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\mime\\__pycache__\\nonmultipart.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\mime\\__pycache__\\text.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\email\\mime\\application.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\mime\\audio.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\mime\\base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\mime\\image.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\mime\\message.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\mime\\multipart.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\mime\\nonmultipart.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\mime\\text.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\parser.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\policy.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\quoprimime.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\email\\utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\html\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\html\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\html\\__pycache__\\entities.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\html\\__pycache__\\parser.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\html\\entities.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\html\\parser.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\http\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\http\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\http\\__pycache__\\client.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\http\\__pycache__\\cookiejar.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\http\\__pycache__\\cookies.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\http\\__pycache__\\server.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\http\\client.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\http\\cookiejar.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\http\\cookies.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\http\\server.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\misc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\socket.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\socketserver.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\test\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\test\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\test\\__pycache__\\pystone.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\test\\__pycache__\\ssl_servers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\test\\__pycache__\\support.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\test\\badcert.pem', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\test\\badkey.pem', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\test\\dh512.pem', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\test\\https_svn_python_org_root.pem', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\test\\keycert.passwd.pem', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\test\\keycert.pem', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\test\\keycert2.pem', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\test\\nokia.pem', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\test\\nullbytecert.pem', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\test\\nullcert.pem', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\test\\pystone.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\test\\sha256.pem', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\test\\ssl_cert.pem', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\test\\ssl_key.passwd.pem', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\test\\ssl_key.pem', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\test\\ssl_servers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\test\\support.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\total_ordering.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\urllib\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\urllib\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\urllib\\__pycache__\\error.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\urllib\\__pycache__\\parse.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\urllib\\__pycache__\\request.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\urllib\\__pycache__\\response.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\urllib\\__pycache__\\robotparser.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\urllib\\error.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\urllib\\parse.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\urllib\\request.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\urllib\\response.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\urllib\\robotparser.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\xmlrpc\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\xmlrpc\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\xmlrpc\\__pycache__\\client.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\xmlrpc\\__pycache__\\server.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\backports\\xmlrpc\\client.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\backports\\xmlrpc\\server.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\builtins\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\builtins\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\builtins\\__pycache__\\disabled.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\builtins\\__pycache__\\iterators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\builtins\\__pycache__\\misc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\builtins\\__pycache__\\new_min_max.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\builtins\\__pycache__\\newnext.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\builtins\\__pycache__\\newround.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\builtins\\__pycache__\\newsuper.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\builtins\\disabled.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\builtins\\iterators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\builtins\\misc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\builtins\\new_min_max.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\builtins\\newnext.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\builtins\\newround.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\builtins\\newsuper.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\__pycache__\\_dummy_thread.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\__pycache__\\_markupbase.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\__pycache__\\_thread.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\__pycache__\\builtins.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\__pycache__\\collections.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\__pycache__\\configparser.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\__pycache__\\copyreg.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\__pycache__\\itertools.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\__pycache__\\multiprocessing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\__pycache__\\pickle.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\__pycache__\\queue.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\__pycache__\\reprlib.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\__pycache__\\socketserver.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\__pycache__\\subprocess.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\__pycache__\\sys.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\__pycache__\\winreg.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\_dummy_thread.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\_markupbase.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\_thread.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\builtins.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\collections.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\configparser.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\copyreg.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\dbm\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\dbm\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\dbm\\__pycache__\\dumb.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\dbm\\__pycache__\\gnu.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\dbm\\__pycache__\\ndbm.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\dbm\\dumb.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\dbm\\gnu.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\dbm\\ndbm.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\html\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\html\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\html\\__pycache__\\entities.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\html\\__pycache__\\parser.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\html\\entities.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\html\\parser.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\http\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\http\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\http\\__pycache__\\client.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\http\\__pycache__\\cookiejar.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\http\\__pycache__\\cookies.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\http\\__pycache__\\server.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\http\\client.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\http\\cookiejar.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\http\\cookies.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\http\\server.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\itertools.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\multiprocessing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\pickle.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\queue.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\reprlib.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\socketserver.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\subprocess.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\sys.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\test\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\test\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\test\\__pycache__\\support.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\test\\support.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\tkinter\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\tkinter\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\tkinter\\__pycache__\\colorchooser.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\tkinter\\__pycache__\\commondialog.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\tkinter\\__pycache__\\constants.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\tkinter\\__pycache__\\dialog.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\tkinter\\__pycache__\\dnd.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\tkinter\\__pycache__\\filedialog.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\tkinter\\__pycache__\\font.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\tkinter\\__pycache__\\messagebox.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\tkinter\\__pycache__\\scrolledtext.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\tkinter\\__pycache__\\simpledialog.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\tkinter\\__pycache__\\tix.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\tkinter\\__pycache__\\ttk.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\tkinter\\colorchooser.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\tkinter\\commondialog.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\tkinter\\constants.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\tkinter\\dialog.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\tkinter\\dnd.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\tkinter\\filedialog.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\tkinter\\font.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\tkinter\\messagebox.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\tkinter\\scrolledtext.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\tkinter\\simpledialog.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\tkinter\\tix.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\tkinter\\ttk.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\urllib\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\urllib\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\urllib\\__pycache__\\error.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\urllib\\__pycache__\\parse.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\urllib\\__pycache__\\request.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\urllib\\__pycache__\\response.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\urllib\\__pycache__\\robotparser.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\urllib\\error.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\urllib\\parse.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\urllib\\request.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\urllib\\response.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\urllib\\robotparser.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\winreg.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\xmlrpc\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\xmlrpc\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\xmlrpc\\__pycache__\\client.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\xmlrpc\\__pycache__\\server.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\moves\\xmlrpc\\client.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\moves\\xmlrpc\\server.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\standard_library\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\standard_library\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\tests\\__pycache__\\base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\tests\\base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\types\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\types\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\types\\__pycache__\\newbytes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\types\\__pycache__\\newdict.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\types\\__pycache__\\newint.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\types\\__pycache__\\newlist.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\types\\__pycache__\\newmemoryview.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\types\\__pycache__\\newobject.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\types\\__pycache__\\newopen.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\types\\__pycache__\\newrange.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\types\\__pycache__\\newstr.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\types\\newbytes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\types\\newdict.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\types\\newint.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\types\\newlist.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\types\\newmemoryview.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\types\\newobject.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\types\\newopen.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\types\\newrange.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\types\\newstr.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\utils\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\future\\utils\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\utils\\__pycache__\\surrogateescape.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\future\\utils\\surrogateescape.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\__pycache__\\_functions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\__pycache__\\_functions_helpers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\__pycache__\\alembic_helpers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\__pycache__\\comparator.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\__pycache__\\elements.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\__pycache__\\exc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\__pycache__\\functions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\__pycache__\\shape.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\__pycache__\\utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\_functions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\_functions_helpers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\admin\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\admin\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\admin\\__pycache__\\plugin.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\admin\\dialects\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\admin\\dialects\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\admin\\dialects\\__pycache__\\common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\admin\\dialects\\__pycache__\\geopackage.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\admin\\dialects\\__pycache__\\mariadb.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\admin\\dialects\\__pycache__\\mysql.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\admin\\dialects\\__pycache__\\postgresql.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\admin\\dialects\\__pycache__\\sqlite.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\admin\\dialects\\common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\admin\\dialects\\geopackage.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\admin\\dialects\\mariadb.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\admin\\dialects\\mysql.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\admin\\dialects\\postgresql.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\admin\\dialects\\sqlite.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\admin\\plugin.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\alembic_helpers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\comparator.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\elements.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\exc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\functions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\functions.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\shape.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\types\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\types\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\types\\dialects\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\types\\dialects\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\types\\dialects\\__pycache__\\common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\types\\dialects\\__pycache__\\geopackage.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\types\\dialects\\__pycache__\\mariadb.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\types\\dialects\\__pycache__\\mysql.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\types\\dialects\\__pycache__\\postgresql.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\types\\dialects\\__pycache__\\sqlite.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoalchemy2\\types\\dialects\\common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\types\\dialects\\geopackage.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\types\\dialects\\mariadb.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\types\\dialects\\mysql.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\types\\dialects\\postgresql.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\types\\dialects\\sqlite.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoalchemy2\\utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas-1.0.1.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas-1.0.1.dist-info\\LICENSE.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas-1.0.1.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas-1.0.1.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas-1.0.1.dist-info\\REQUESTED', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas-1.0.1.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas-1.0.1.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\__pycache__\\_compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\__pycache__\\_config.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\__pycache__\\_decorator.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\__pycache__\\_version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\__pycache__\\array.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\__pycache__\\base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\__pycache__\\explore.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\__pycache__\\geodataframe.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\__pycache__\\geoseries.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\__pycache__\\plotting.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\__pycache__\\sindex.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\__pycache__\\testing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\_compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\_config.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\_decorator.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\_version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\array.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\datasets\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\datasets\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\explore.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\geodataframe.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\geoseries.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\io\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\io\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\io\\__pycache__\\_geoarrow.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\io\\__pycache__\\_pyarrow_hotfix.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\io\\__pycache__\\arrow.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\io\\__pycache__\\file.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\io\\__pycache__\\sql.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\io\\__pycache__\\util.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\io\\_geoarrow.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\io\\_pyarrow_hotfix.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\io\\arrow.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\io\\file.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\io\\sql.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\io\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\io\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\io\\tests\\__pycache__\\generate_legacy_storage_files.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\io\\tests\\__pycache__\\test_arrow.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\io\\tests\\__pycache__\\test_file.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\io\\tests\\__pycache__\\test_file_geom_types_drivers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\io\\tests\\__pycache__\\test_geoarrow.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\io\\tests\\__pycache__\\test_infer_schema.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\io\\tests\\__pycache__\\test_pickle.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\io\\tests\\__pycache__\\test_sql.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\io\\tests\\generate_legacy_storage_files.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\io\\tests\\test_arrow.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\io\\tests\\test_file.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\io\\tests\\test_file_geom_types_drivers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\io\\tests\\test_geoarrow.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\io\\tests\\test_infer_schema.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\io\\tests\\test_pickle.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\io\\tests\\test_sql.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\io\\util.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\plotting.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\sindex.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\testing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_array.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_config.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_crs.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_datasets.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_decorator.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_dissolve.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_explore.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_extension_array.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_geocode.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_geodataframe.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_geom_methods.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_geoseries.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_merge.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_op_output_types.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_overlay.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_pandas_methods.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_plotting.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_show_versions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_sindex.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_testing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\test_types.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\__pycache__\\util.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tests\\data\\null_geom.geojson', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_array.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_config.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_crs.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_datasets.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_decorator.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_dissolve.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_explore.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_extension_array.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_geocode.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_geodataframe.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_geom_methods.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_geoseries.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_merge.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_op_output_types.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_overlay.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_pandas_methods.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_plotting.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_show_versions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_sindex.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_testing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\test_types.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tests\\util.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tools\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tools\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tools\\__pycache__\\_random.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tools\\__pycache__\\_show_versions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tools\\__pycache__\\clip.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tools\\__pycache__\\geocoding.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tools\\__pycache__\\hilbert_curve.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tools\\__pycache__\\overlay.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tools\\__pycache__\\sjoin.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tools\\__pycache__\\util.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tools\\_random.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tools\\_show_versions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tools\\clip.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tools\\geocoding.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tools\\hilbert_curve.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tools\\overlay.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tools\\sjoin.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tools\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tools\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tools\\tests\\__pycache__\\test_clip.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tools\\tests\\__pycache__\\test_hilbert_curve.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tools\\tests\\__pycache__\\test_random.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tools\\tests\\__pycache__\\test_sjoin.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tools\\tests\\__pycache__\\test_tools.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geopandas\\tools\\tests\\test_clip.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tools\\tests\\test_hilbert_curve.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tools\\tests\\test_random.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tools\\tests\\test_sjoin.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tools\\tests\\test_tools.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geopandas\\tools\\util.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoserver\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoserver\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoserver\\__pycache__\\catalog.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoserver\\__pycache__\\layer.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoserver\\__pycache__\\layergroup.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoserver\\__pycache__\\namespace.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoserver\\__pycache__\\resource.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoserver\\__pycache__\\security.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoserver\\__pycache__\\service.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoserver\\__pycache__\\settings.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoserver\\__pycache__\\store.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoserver\\__pycache__\\style.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoserver\\__pycache__\\support.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoserver\\__pycache__\\util.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoserver\\__pycache__\\workspace.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\geoserver\\catalog.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoserver\\layer.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoserver\\layergroup.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoserver\\namespace.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoserver\\resource.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoserver\\security.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoserver\\service.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoserver\\settings.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoserver\\store.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoserver\\style.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoserver\\support.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoserver\\util.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoserver\\workspace.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoserver_restconfig-2.0.13.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoserver_restconfig-2.0.13.dist-info\\LICENSE.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoserver_restconfig-2.0.13.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoserver_restconfig-2.0.13.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoserver_restconfig-2.0.13.dist-info\\REQUESTED', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoserver_restconfig-2.0.13.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\geoserver_restconfig-2.0.13.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata-0.5.4.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata-0.5.4.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata-0.5.4.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata-0.5.4.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata-0.5.4.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\both\\good\\sangis.org\\Airport\\Air_Runways.dbf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\both\\good\\sangis.org\\Airport\\Air_Runways.pdf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\both\\good\\sangis.org\\Airport\\Air_Runways.prj', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\both\\good\\sangis.org\\Airport\\Air_Runways.sbn', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\both\\good\\sangis.org\\Airport\\Air_Runways.sbx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\both\\good\\sangis.org\\Airport\\Air_Runways.shp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\both\\good\\sangis.org\\Airport\\Air_Runways.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\both\\good\\sangis.org\\Airport\\Air_Runways.shx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\bad\\grid_without_projection.asc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\bad\\points_epsg2249_no_prj.dbf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\bad\\points_epsg2249_no_prj.shp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\bad\\points_epsg2249_no_prj.shx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\bad\\points_wgs84_no_prj.dbf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\bad\\points_wgs84_no_prj.shp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\bad\\points_wgs84_no_prj.shx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\bad\\unsupported_ext.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\raster\\relief_san_andres.tif', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\raster\\relief_san_andres.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\raster\\test_grid.tif', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\raster\\test_grid.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_date.dbf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_date.prj', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_date.qix', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_date.shp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_date.shx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_date_iso_date.dbf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_date_iso_date.prj', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_date_iso_date.shp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_date_iso_date.shx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_dates_bc.dbf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_dates_bc.prj', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_dates_bc.shp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_dates_bc.shx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_end_date.dbf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_end_date.prj', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_end_date.shp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_end_date.shx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_year_field.dbf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_year_field.prj', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_year_field.shp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_year_field.shx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_year_field2.dbf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_year_field2.prj', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_year_field2.shp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\time\\boxes_with_year_field2.shx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\readme.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_administrative.dbf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_administrative.prj', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_administrative.shp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_administrative.shx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_coastline.dbf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_coastline.prj', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_coastline.shp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_coastline.shx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_highway.dbf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_highway.prj', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_highway.shp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_highway.shx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_location.dbf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_location.prj', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_location.shp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_location.shx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_natural.dbf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_natural.prj', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_natural.shp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_natural.shx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_poi.dbf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_poi.prj', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_poi.shp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_poi.shx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_water.dbf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_water.prj', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_water.shp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\san_andres_y_providencia_water.shx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\single_point.dbf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\single_point.prj', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\single_point.shp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\single_point.shx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\data\\good\\vector\\single_point.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Agriculture\\AG_PRESERVE.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Agriculture\\AG_PRESERVE_CONTRACTS.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Agriculture\\Agriculture_Commodity.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Agriculture\\Agriculture_Point_Of_Sale.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Agriculture\\Agriculture_Standards_Devices.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Agriculture\\Agriculture_Trap_Grid.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Agriculture\\Agriculture_Trap_Sub_Grid.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Agriculture\\Nursery_Locations.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Business\\Business_Enterprise_Zones_SD.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Business\\Business_Improvement_Districts_SD.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Business\\Business_Metro_Revloan.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Business\\Business_Regional_Enterprise_Zones_SD.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Business\\Business_Renewal_Community.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Business\\Business_SD_Revloan.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Business\\Business_Sites.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Business\\Construction_Demo_Recycle.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Business\\Property_Based_Improvement_Districts.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Business\\Recycling_Market_Dev_Zones.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Census\\Census_Blockgroup_Income_Edu.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Census\\Census_Blockgroup_Pop_Housing.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Ecology\\Draft_North_County_MSCP_Version_8.0_Categories.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Ecology\\Environmentally_Sensitive_Areas.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Ecology\\Forest_Conservation_Init.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Ecology\\MSCP_CN.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Ecology\\MSCP_EAST_DRAFT_CN.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Ecology\\MSCP_Habitrak.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Ecology\\Vegetation_CN.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Ecology\\Water_Quality_Sensitive_Areas_SD.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Facilities\\Hospitals.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Facilities\\Law_Enforcement_Facilities.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Facilities\\Library.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Facilities\\Military_Facility.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Facilities\\Waste_Facilities.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Fire\\FIRESTORM_2007_PERIMETERS.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Fire\\FIRE_HISTORY_THRU_2007.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Fire\\Fire_Battalion_Districts.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Fire\\Fire_Burn_History.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Fire\\Fire_Engine_Districts.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Fire\\Fire_Hazard_Severity_Zones.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Fire\\Fire_Stations_CN.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Fire\\Firesafe_Councils.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Fire\\Firestorm_2007_Assist_Center.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Fire\\Firestorm_2007_Damages_CN_All.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Hydrology\\Aquifers.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Hydrology\\Flood_Panel.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Hydrology\\Flood_Plain.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Hydrology\\Lakes.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Parcels_Lots\\Assessor_Book.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Parcels_Lots\\Landbase_Boundaries.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Parcels_Lots\\Lots.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Parcels_Lots\\Parcels.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Parcels_Lots\\Parcels_East.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Parcels_Lots\\Parcels_North.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Parcels_Lots\\Parcels_South.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Parks\\Parks_Active_Use.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Parks\\Parks_CN.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Parks\\Parks_SD.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Places\\Casinos.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Places\\Cell_Phone_Towers_INC.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Places\\Cell_Phone_Towers_UNINC.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Places\\GAS_STATIONS.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Slopes\\Slopes_1992.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Slopes\\Slopes_1999.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Slopes\\Slopes_CN.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Subdivision\\Subdivision.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Subdivision\\Subdivision_Line.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Topo\\Topo_1992_2Ft.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Topo\\Topo_1992_5Ft.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Topo\\Topo_1999_2_North.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Topo\\Topo_1999_2_South.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Topo\\Topo_1999_5.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\sangis.org\\Topo\\Topo_20Ft.shp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIERA.EDGAR-HTAP.so2.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.OMI_NO2.NO2.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.OMI_NO2.NO2Trop.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.OMI_O3.CloudFraction.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.OMI_O3.O3.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.88304.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.AGf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.ALf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.ASf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.BAf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.BRf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.CAf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.CDf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.CHLf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.CLf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.CM_calculated.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.COf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.CRf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.CSf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.CUf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.EC1f.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.EC2f.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.EC3f.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.ECf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.ECf_NIOSH.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.FEf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.HGf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.Hf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.Kf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.MF.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.MGf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.MNf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.MOf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.MT.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.N2f.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.NAf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.NH4f.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.NIf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.NO3f.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.OC1f.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.OC2f.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.OC3f.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.OC4f.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.OCf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.OCf_NIOSH.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.OMC1.8.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.OMCf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.OPf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.PBf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.Pf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.RCFM.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.SEf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.SIf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.SO4f.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.SOILf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.SRf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.SeaSalt.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.Sf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.TIf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.Vf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.ZNf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.ZRf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.ammNO3f.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.ammSO4f.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.dv.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.fabs.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.nv_nitrate.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.v_nitrate.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\CIRA.VIEWS.wolf.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.mmr_bc_hydrophilic.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.mmr_bc_hydrophobic.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.mmr_dust1.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.mmr_dust2.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.mmr_dust3.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.mmr_oc_hydrophilic.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.mmr_oc_hydrophobic.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.mmr_seasalt1.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.mmr_seasalt2.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.mmr_seasalt3.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.mmr_sulfaer.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.ps.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.temp.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.vmr_bigalk.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.vmr_bigene.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.vmr_c2h6.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.vmr_ch2o.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.vmr_ch3cho.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.vmr_ch4.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.vmr_co.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.vmr_hno3.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.vmr_ho2no2.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.vmr_isop.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.vmr_n2o.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.vmr_no.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.vmr_no2.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.vmr_o3.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.vmr_pan.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.vmr_so2.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\gisdata\\metadata\\good\\wustl.edu\\FZ Juelich.MACC.vmr_toluene.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet-3.1.1.dist-info\\AUTHORS', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet-3.1.1.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet-3.1.1.dist-info\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet-3.1.1.dist-info\\LICENSE.PSF', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet-3.1.1.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet-3.1.1.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet-3.1.1.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet-3.1.1.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\CObjects.cpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\PyGreenlet.cpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\PyGreenlet.hpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\PyGreenletUnswitchable.cpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\PyModule.cpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\TBrokenGreenlet.cpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\TExceptionState.cpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\TGreenlet.cpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\TGreenlet.hpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\TGreenletGlobals.cpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\TMainGreenlet.cpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\TPythonState.cpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\TStackState.cpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\TThreadState.hpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\TThreadStateCreator.hpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\TThreadStateDestroy.cpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\TUserGreenlet.cpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\_greenlet.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\greenlet.cpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\greenlet.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\greenlet_allocator.hpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\greenlet_compiler_compat.hpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\greenlet_cpython_add_pending.hpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\greenlet_cpython_compat.hpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\greenlet_exceptions.hpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\greenlet_internal.hpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\greenlet_refs.hpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\greenlet_slp_switch.hpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\greenlet_thread_support.hpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\platform\\setup_switch_x64_masm.cmd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_aarch64_gcc.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_alpha_unix.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_amd64_unix.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_arm32_gcc.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_arm32_ios.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_arm64_masm.asm', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_arm64_masm.obj', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\platform\\switch_arm64_msvc.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_csky_gcc.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_loongarch64_linux.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_m68k_gcc.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_mips_unix.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_ppc64_aix.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_ppc64_linux.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_ppc_aix.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_ppc_linux.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_ppc_macosx.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_ppc_unix.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_riscv_unix.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_s390_unix.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_sh_gcc.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_sparc_sun_gcc.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_x32_unix.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_x64_masm.asm', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_x64_masm.obj', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\platform\\switch_x64_msvc.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_x86_msvc.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\platform\\switch_x86_unix.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\slp_platformselect.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\fail_clearing_run_switches.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\fail_cpp_exception.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\fail_initialstub_already_started.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\fail_slp_switch.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\fail_switch_three_greenlets.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\fail_switch_three_greenlets2.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\fail_switch_two_greenlets.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\leakcheck.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\test_contextvars.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\test_cpp.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\test_extension_interface.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\test_gc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\test_generator.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\test_generator_nested.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\test_greenlet.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\test_greenlet_trash.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\test_leaks.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\test_stack_saved.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\test_throw.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\test_tracing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\test_version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\__pycache__\\test_weakref.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\_test_extension.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\_test_extension.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\_test_extension_cpp.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\greenlet\\tests\\_test_extension_cpp.cpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\fail_clearing_run_switches.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\fail_cpp_exception.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\fail_initialstub_already_started.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\fail_slp_switch.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\fail_switch_three_greenlets.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\fail_switch_three_greenlets2.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\fail_switch_two_greenlets.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\leakcheck.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\test_contextvars.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\test_cpp.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\test_extension_interface.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\test_gc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\test_generator.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\test_generator_nested.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\test_greenlet.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\test_greenlet_trash.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\test_leaks.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\test_stack_saved.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\test_throw.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\test_tracing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\test_version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\greenlet\\tests\\test_weakref.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\idna-3.10.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\idna-3.10.dist-info\\LICENSE.md', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\idna-3.10.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\idna-3.10.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\idna-3.10.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\idna\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\idna\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\idna\\__pycache__\\codec.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\idna\\__pycache__\\compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\idna\\__pycache__\\core.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\idna\\__pycache__\\idnadata.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\idna\\__pycache__\\intranges.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\idna\\__pycache__\\package_data.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\idna\\__pycache__\\uts46data.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\idna\\codec.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\idna\\compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\idna\\core.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\idna\\idnadata.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\idna\\intranges.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\idna\\package_data.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\idna\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\idna\\uts46data.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\__pycache__\\fixer_util.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\__pycache__\\main.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixer_util.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_UserDict.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_absolute_import.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_add__future__imports_except_unicode_literals.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_basestring.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_bytes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_cmp.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_division.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_division_safe.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_execfile.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_future_builtins.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_future_standard_library.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_future_standard_library_urllib.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_input.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_metaclass.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_next_call.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_object.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_oldstr_wrap.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_order___future__imports.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_print.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_print_with_import.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_raise.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_remove_old__future__imports.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_unicode_keep_u.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_unicode_literals_import.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\__pycache__\\fix_xrange_with_import.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_UserDict.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_absolute_import.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_add__future__imports_except_unicode_literals.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_basestring.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_bytes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_cmp.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_division.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_division_safe.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_execfile.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_future_builtins.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_future_standard_library.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_future_standard_library_urllib.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_input.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_metaclass.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_next_call.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_object.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_oldstr_wrap.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_order___future__imports.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_print.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_print_with_import.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_raise.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_remove_old__future__imports.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_unicode_keep_u.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_unicode_literals_import.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\fixes\\fix_xrange_with_import.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libfuturize\\main.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\__pycache__\\main.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\feature_base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\fix_add_all__future__imports.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\fix_add_all_future_builtins.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\fix_add_future_standard_library_import.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\fix_annotations.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\fix_division.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\fix_features.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\fix_fullargspec.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\fix_future_builtins.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\fix_getcwd.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\fix_imports.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\fix_imports2.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\fix_kwargs.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\fix_memoryview.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\fix_metaclass.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\fix_newstyle.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\fix_next.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\fix_printfunction.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\fix_raise.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\fix_raise_.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\fix_throw.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\__pycache__\\fix_unpacking.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\libpasteurize\\fixes\\feature_base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\fixes\\fix_add_all__future__imports.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\fixes\\fix_add_all_future_builtins.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\fixes\\fix_add_future_standard_library_import.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\fixes\\fix_annotations.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\fixes\\fix_division.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\fixes\\fix_features.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\fixes\\fix_fullargspec.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\fixes\\fix_future_builtins.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\fixes\\fix_getcwd.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\fixes\\fix_imports.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\fixes\\fix_imports2.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\fixes\\fix_kwargs.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\fixes\\fix_memoryview.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\fixes\\fix_metaclass.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\fixes\\fix_newstyle.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\fixes\\fix_next.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\fixes\\fix_printfunction.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\fixes\\fix_raise.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\fixes\\fix_raise_.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\fixes\\fix_throw.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\fixes\\fix_unpacking.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\libpasteurize\\main.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy-2.2.3-cp310-cp310-win_amd64.whl', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy-2.2.3.dist-info\\DELVEWHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy-2.2.3.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy-2.2.3.dist-info\\LICENSE.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy-2.2.3.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy-2.2.3.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy-2.2.3.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy-2.2.3.dist-info\\entry_points.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy.libs\\libscipy_openblas64_-43e11ff0749b8cbe0a615c9cf6737e0e.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy.libs\\msvcp140-263139962577ecda4cd9469ca360a746.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\__config__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\__config__.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\__init__.cython-30.pxd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\__init__.pxd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\__init__.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\__pycache__\\__config__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\__pycache__\\_array_api_info.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\__pycache__\\_configtool.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\__pycache__\\_distributor_init.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\__pycache__\\_expired_attrs_2_0.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\__pycache__\\_globals.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\__pycache__\\_pytesttester.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\__pycache__\\ctypeslib.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\__pycache__\\dtypes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\__pycache__\\exceptions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\__pycache__\\matlib.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\__pycache__\\version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_array_api_info.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_array_api_info.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_configtool.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_configtool.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\__init__.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\_add_newdocs.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\_add_newdocs_scalars.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\_asarray.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\_dtype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\_dtype_ctypes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\_exceptions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\_internal.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\_machar.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\_methods.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\_string_helpers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\_type_aliases.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\_ufunc_config.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\arrayprint.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\cversions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\defchararray.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\einsumfunc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\fromnumeric.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\function_base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\getlimits.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\memmap.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\multiarray.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\numeric.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\numerictypes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\overrides.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\printoptions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\records.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\shape_base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\strings.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\__pycache__\\umath.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\_add_newdocs.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\_add_newdocs_scalars.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\_asarray.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\_asarray.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\_dtype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\_dtype_ctypes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\_exceptions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\_internal.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\_internal.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\_machar.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\_methods.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\_multiarray_tests.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\_multiarray_tests.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\_multiarray_umath.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\_multiarray_umath.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\_operand_flag_tests.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\_operand_flag_tests.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\_rational_tests.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\_rational_tests.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\_simd.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\_simd.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\_string_helpers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\_struct_ufunc_tests.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\_struct_ufunc_tests.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\_type_aliases.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\_type_aliases.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\_ufunc_config.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\_ufunc_config.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\_umath_tests.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\_umath_tests.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\arrayprint.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\arrayprint.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\cversions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\defchararray.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\defchararray.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\einsumfunc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\einsumfunc.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\fromnumeric.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\fromnumeric.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\function_base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\function_base.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\getlimits.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\getlimits.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\__multiarray_api.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\__multiarray_api.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\__ufunc_api.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\__ufunc_api.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\_neighborhood_iterator_imp.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\_numpyconfig.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\_public_dtype_api_table.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\arrayobject.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\arrayscalars.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\dtype_api.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\halffloat.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\ndarrayobject.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\ndarraytypes.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\npy_1_7_deprecated_api.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\npy_2_compat.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\npy_2_complexcompat.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\npy_3kcompat.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\npy_common.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\npy_cpu.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\npy_endian.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\npy_math.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\npy_no_deprecated_api.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\npy_os.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\numpyconfig.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\random\\LICENSE.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\random\\bitgen.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\random\\distributions.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\random\\libdivide.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\ufuncobject.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\include\\numpy\\utils.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\lib\\npy-pkg-config\\mlib.ini', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\lib\\npy-pkg-config\\npymath.ini', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\lib\\npymath.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\lib\\pkgconfig\\numpy.pc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\memmap.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\memmap.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\multiarray.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\multiarray.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\numeric.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\numeric.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\numerictypes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\numerictypes.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\overrides.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\printoptions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\records.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\records.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\shape_base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\shape_base.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\strings.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\strings.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\_locales.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\_natype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test__exceptions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_abc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_argparse.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_array_api_info.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_array_coercion.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_array_interface.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_arraymethod.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_arrayobject.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_arrayprint.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_casting_floatingpoint_errors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_casting_unittests.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_conversion_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_cpu_dispatcher.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_cpu_features.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_custom_dtypes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_cython.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_datetime.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_defchararray.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_deprecations.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_dlpack.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_dtype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_einsum.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_errstate.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_extint128.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_function_base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_getlimits.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_half.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_hashtable.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_indexerrors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_item_selection.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_limited_api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_longdouble.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_machar.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_mem_overlap.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_mem_policy.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_memmap.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_multiarray.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_multithreading.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_nditer.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_nep50_promotions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_numeric.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_numerictypes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_overrides.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_print.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_protocols.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_records.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_regression.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_scalar_ctors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_scalar_methods.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_scalarbuffer.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_scalarinherit.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_scalarmath.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_scalarprint.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_shape_base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_simd.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_simd_module.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_stringdtype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_strings.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_ufunc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_umath.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_umath_accuracy.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_umath_complex.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\__pycache__\\test_unicode.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\_locales.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\_natype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\astype_copy.pkl', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\generate_umath_validation_data.cpp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\recarray_from_file.fits', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\umath-validation-set-README.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\umath-validation-set-arccos.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\umath-validation-set-arccosh.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\umath-validation-set-arcsin.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\umath-validation-set-arcsinh.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\umath-validation-set-arctan.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\umath-validation-set-arctanh.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\umath-validation-set-cbrt.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\umath-validation-set-cos.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\umath-validation-set-cosh.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\umath-validation-set-exp.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\umath-validation-set-exp2.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\umath-validation-set-expm1.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\umath-validation-set-log.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\umath-validation-set-log10.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\umath-validation-set-log1p.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\umath-validation-set-log2.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\umath-validation-set-sin.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\umath-validation-set-sinh.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\umath-validation-set-tan.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\data\\umath-validation-set-tanh.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\examples\\cython\\__pycache__\\setup.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\examples\\cython\\checks.pyx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\examples\\cython\\meson.build', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\examples\\cython\\setup.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\examples\\limited_api\\__pycache__\\setup.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_core\\tests\\examples\\limited_api\\limited_api1.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\examples\\limited_api\\limited_api2.pyx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\examples\\limited_api\\limited_api_latest.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\examples\\limited_api\\meson.build', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\examples\\limited_api\\setup.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test__exceptions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_abc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_argparse.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_array_api_info.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_array_coercion.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_array_interface.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_arraymethod.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_arrayobject.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_arrayprint.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_casting_floatingpoint_errors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_casting_unittests.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_conversion_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_cpu_dispatcher.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_cpu_features.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_custom_dtypes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_cython.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_datetime.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_defchararray.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_deprecations.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_dlpack.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_dtype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_einsum.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_errstate.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_extint128.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_function_base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_getlimits.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_half.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_hashtable.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_indexerrors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_item_selection.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_limited_api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_longdouble.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_machar.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_mem_overlap.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_mem_policy.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_memmap.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_multiarray.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_multithreading.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_nditer.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_nep50_promotions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_numeric.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_numerictypes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_overrides.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_print.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_protocols.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_records.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_regression.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_scalar_ctors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_scalar_methods.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_scalarbuffer.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_scalarinherit.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_scalarmath.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_scalarprint.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_shape_base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_simd.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_simd_module.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_stringdtype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_strings.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_ufunc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_umath.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_umath_accuracy.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_umath_complex.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\tests\\test_unicode.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_core\\umath.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_distributor_init.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_distributor_init.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_expired_attrs_2_0.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_expired_attrs_2_0.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_globals.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_globals.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_pyinstaller\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_pyinstaller\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_pyinstaller\\__pycache__\\hook-numpy.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_pyinstaller\\hook-numpy.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_pyinstaller\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_pyinstaller\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_pyinstaller\\tests\\__pycache__\\pyinstaller-smoke.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_pyinstaller\\tests\\__pycache__\\test_pyinstaller.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_pyinstaller\\tests\\pyinstaller-smoke.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_pyinstaller\\tests\\test_pyinstaller.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_pytesttester.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_pytesttester.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_typing\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_typing\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_typing\\__pycache__\\_add_docstring.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_typing\\__pycache__\\_array_like.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_typing\\__pycache__\\_char_codes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_typing\\__pycache__\\_dtype_like.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_typing\\__pycache__\\_extended_precision.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_typing\\__pycache__\\_nbit.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_typing\\__pycache__\\_nbit_base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_typing\\__pycache__\\_nested_sequence.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_typing\\__pycache__\\_scalars.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_typing\\__pycache__\\_shape.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_typing\\__pycache__\\_ufunc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_typing\\_add_docstring.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_typing\\_array_like.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_typing\\_callable.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_typing\\_char_codes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_typing\\_dtype_like.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_typing\\_extended_precision.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_typing\\_nbit.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_typing\\_nbit_base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_typing\\_nested_sequence.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_typing\\_scalars.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_typing\\_shape.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_typing\\_ufunc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_typing\\_ufunc.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_utils\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_utils\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_utils\\__pycache__\\_convertions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_utils\\__pycache__\\_inspect.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_utils\\__pycache__\\_pep440.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\_utils\\_convertions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_utils\\_inspect.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\_utils\\_pep440.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\char\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\char\\__init__.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\char\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\compat\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\compat\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\compat\\__pycache__\\py3k.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\compat\\py3k.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\compat\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\compat\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\core\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\core\\__init__.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\core\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\core\\__pycache__\\_dtype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\core\\__pycache__\\_dtype_ctypes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\core\\__pycache__\\_internal.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\core\\__pycache__\\_multiarray_umath.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\core\\__pycache__\\_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\core\\__pycache__\\arrayprint.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\core\\__pycache__\\defchararray.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\core\\__pycache__\\einsumfunc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\core\\__pycache__\\fromnumeric.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\core\\__pycache__\\function_base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\core\\__pycache__\\getlimits.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\core\\__pycache__\\multiarray.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\core\\__pycache__\\numeric.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\core\\__pycache__\\numerictypes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\core\\__pycache__\\overrides.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\core\\__pycache__\\records.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\core\\__pycache__\\shape_base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\core\\__pycache__\\umath.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\core\\_dtype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\core\\_dtype_ctypes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\core\\_internal.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\core\\_multiarray_umath.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\core\\_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\core\\arrayprint.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\core\\defchararray.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\core\\einsumfunc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\core\\fromnumeric.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\core\\function_base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\core\\getlimits.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\core\\multiarray.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\core\\numeric.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\core\\numerictypes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\core\\overrides.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\core\\records.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\core\\shape_base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\core\\umath.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ctypeslib.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ctypeslib.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\__init__.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\_shell_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\armccompiler.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\ccompiler.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\ccompiler_opt.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\conv_template.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\core.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\cpuinfo.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\exec_command.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\extension.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\from_template.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\fujitsuccompiler.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\intelccompiler.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\lib2def.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\line_endings.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\log.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\mingw32ccompiler.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\misc_util.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\msvc9compiler.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\msvccompiler.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\npy_pkg_config.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\numpy_distribution.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\pathccompiler.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\system_info.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\__pycache__\\unixccompiler.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\_shell_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\armccompiler.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\ccompiler.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\ccompiler_opt.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_asimd.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_asimddp.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_asimdfhm.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_asimdhp.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_avx.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_avx2.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_avx512_clx.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_avx512_cnl.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_avx512_icl.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_avx512_knl.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_avx512_knm.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_avx512_skx.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_avx512_spr.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_avx512cd.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_avx512f.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_f16c.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_fma3.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_fma4.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_neon.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_neon_fp16.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_neon_vfpv4.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_popcnt.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_rvv.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_sse.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_sse2.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_sse3.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_sse41.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_sse42.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_ssse3.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_sve.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_vsx.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_vsx2.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_vsx3.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_vsx4.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_vx.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_vxe.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_vxe2.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\cpu_xop.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\extra_avx512bw_mask.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\extra_avx512dq_mask.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\extra_avx512f_reduce.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\extra_vsx3_half_double.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\extra_vsx4_mma.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\extra_vsx_asm.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\checks\\test_flags.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\command\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\command\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\command\\__pycache__\\autodist.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\command\\__pycache__\\bdist_rpm.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\command\\__pycache__\\build.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\command\\__pycache__\\build_clib.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\command\\__pycache__\\build_ext.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\command\\__pycache__\\build_py.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\command\\__pycache__\\build_scripts.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\command\\__pycache__\\build_src.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\command\\__pycache__\\config.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\command\\__pycache__\\config_compiler.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\command\\__pycache__\\develop.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\command\\__pycache__\\egg_info.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\command\\__pycache__\\install.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\command\\__pycache__\\install_clib.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\command\\__pycache__\\install_data.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\command\\__pycache__\\install_headers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\command\\__pycache__\\sdist.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\command\\autodist.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\command\\bdist_rpm.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\command\\build.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\command\\build_clib.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\command\\build_ext.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\command\\build_py.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\command\\build_scripts.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\command\\build_src.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\command\\config.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\command\\config_compiler.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\command\\develop.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\command\\egg_info.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\command\\install.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\command\\install_clib.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\command\\install_data.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\command\\install_headers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\command\\sdist.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\conv_template.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\core.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\cpuinfo.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\exec_command.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\extension.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\__pycache__\\absoft.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\__pycache__\\arm.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\__pycache__\\compaq.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\__pycache__\\environment.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\__pycache__\\fujitsu.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\__pycache__\\g95.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\__pycache__\\gnu.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\__pycache__\\hpux.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\__pycache__\\ibm.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\__pycache__\\intel.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\__pycache__\\lahey.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\__pycache__\\mips.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\__pycache__\\nag.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\__pycache__\\none.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\__pycache__\\nv.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\__pycache__\\pathf95.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\__pycache__\\pg.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\__pycache__\\sun.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\__pycache__\\vast.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\absoft.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\arm.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\compaq.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\environment.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\fujitsu.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\g95.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\gnu.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\hpux.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\ibm.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\intel.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\lahey.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\mips.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\nag.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\none.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\nv.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\pathf95.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\pg.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\sun.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\fcompiler\\vast.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\from_template.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\fujitsuccompiler.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\intelccompiler.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\lib2def.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\line_endings.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\log.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\mingw32ccompiler.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\mingw\\gfortran_vs2003_hack.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\misc_util.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\msvc9compiler.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\msvccompiler.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\npy_pkg_config.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\numpy_distribution.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\pathccompiler.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\system_info.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\tests\\__pycache__\\test_build_ext.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\tests\\__pycache__\\test_ccompiler_opt.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\tests\\__pycache__\\test_ccompiler_opt_conf.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\tests\\__pycache__\\test_exec_command.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\tests\\__pycache__\\test_fcompiler.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\tests\\__pycache__\\test_fcompiler_gnu.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\tests\\__pycache__\\test_fcompiler_intel.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\tests\\__pycache__\\test_fcompiler_nagfor.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\tests\\__pycache__\\test_from_template.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\tests\\__pycache__\\test_log.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\tests\\__pycache__\\test_mingw32ccompiler.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\tests\\__pycache__\\test_misc_util.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\tests\\__pycache__\\test_npy_pkg_config.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\tests\\__pycache__\\test_shell_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\tests\\__pycache__\\test_system_info.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\tests\\__pycache__\\utilities.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\distutils\\tests\\test_build_ext.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\tests\\test_ccompiler_opt.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\tests\\test_ccompiler_opt_conf.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\tests\\test_exec_command.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\tests\\test_fcompiler.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\tests\\test_fcompiler_gnu.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\tests\\test_fcompiler_intel.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\tests\\test_fcompiler_nagfor.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\tests\\test_from_template.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\tests\\test_log.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\tests\\test_mingw32ccompiler.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\tests\\test_misc_util.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\tests\\test_npy_pkg_config.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\tests\\test_shell_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\tests\\test_system_info.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\tests\\utilities.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\distutils\\unixccompiler.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\doc\\__pycache__\\ufuncs.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\doc\\ufuncs.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\dtypes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\dtypes.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\exceptions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\exceptions.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\__init__.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\__main__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\__pycache__\\__main__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\__pycache__\\__version__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\__pycache__\\_isocbind.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\__pycache__\\_src_pyf.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\__pycache__\\auxfuncs.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\__pycache__\\capi_maps.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\__pycache__\\cb_rules.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\__pycache__\\cfuncs.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\__pycache__\\common_rules.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\__pycache__\\crackfortran.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\__pycache__\\diagnose.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\__pycache__\\f2py2e.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\__pycache__\\f90mod_rules.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\__pycache__\\func2subr.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\__pycache__\\rules.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\__pycache__\\symbolic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\__pycache__\\use_rules.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\__version__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\_backends\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\_backends\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\_backends\\__pycache__\\_backend.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\_backends\\__pycache__\\_distutils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\_backends\\__pycache__\\_meson.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\_backends\\_backend.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\_backends\\_distutils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\_backends\\_meson.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\_backends\\meson.build.template', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\_isocbind.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\_src_pyf.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\auxfuncs.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\capi_maps.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\cb_rules.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\cfuncs.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\common_rules.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\crackfortran.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\diagnose.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\f2py2e.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\f90mod_rules.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\func2subr.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\rules.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\setup.cfg', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\src\\fortranobject.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\src\\fortranobject.h', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\symbolic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_abstract_interface.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_array_from_pyobj.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_assumed_shape.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_block_docstring.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_callback.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_character.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_crackfortran.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_data.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_docs.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_f2cmap.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_f2py2e.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_isoc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_kind.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_mixed.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_modules.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_parameter.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_pyf_src.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_quoted_character.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_regression.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_return_character.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_return_complex.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_return_integer.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_return_logical.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_return_real.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_routines.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_semicolon_split.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_size.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_string.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_symbolic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\test_value_attrspec.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\__pycache__\\util.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\abstract_interface\\foo.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\abstract_interface\\gh18403_mod.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\array_from_pyobj\\wrapmodule.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\assumed_shape\\.f2py_f2cmap', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\assumed_shape\\foo_free.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\assumed_shape\\foo_mod.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\assumed_shape\\foo_use.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\assumed_shape\\precision.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\block_docstring\\foo.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\callback\\foo.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\callback\\gh17797.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\callback\\gh18335.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\callback\\gh25211.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\callback\\gh25211.pyf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\callback\\gh26681.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\cli\\gh_22819.pyf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\cli\\hi77.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\cli\\hiworld.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\common\\block.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\common\\gh19161.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\crackfortran\\accesstype.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\crackfortran\\data_common.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\crackfortran\\data_multiplier.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\crackfortran\\data_stmts.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\crackfortran\\data_with_comments.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\crackfortran\\foo_deps.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\crackfortran\\gh15035.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\crackfortran\\gh17859.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\crackfortran\\gh22648.pyf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\crackfortran\\gh23533.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\crackfortran\\gh23598.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\crackfortran\\gh23598Warn.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\crackfortran\\gh23879.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\crackfortran\\gh27697.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\crackfortran\\gh2848.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\crackfortran\\operators.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\crackfortran\\privatemod.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\crackfortran\\publicmod.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\crackfortran\\pubprivmod.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\crackfortran\\unicode_comment.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\f2cmap\\.f2py_f2cmap', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\f2cmap\\isoFortranEnvMap.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\isocintrin\\isoCtests.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\kind\\foo.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\mixed\\foo.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\mixed\\foo_fixed.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\mixed\\foo_free.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\modules\\gh25337\\data.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\modules\\gh25337\\use_data.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\modules\\gh26920\\two_mods_with_no_public_entities.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\modules\\gh26920\\two_mods_with_one_public_routine.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\modules\\module_data_docstring.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\modules\\use_modules.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\negative_bounds\\issue_20853.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\parameter\\constant_array.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\parameter\\constant_both.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\parameter\\constant_compound.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\parameter\\constant_integer.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\parameter\\constant_non_compound.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\parameter\\constant_real.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\quoted_character\\foo.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\regression\\AB.inc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\regression\\assignOnlyModule.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\regression\\datonly.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\regression\\f77comments.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\regression\\f77fixedform.f95', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\regression\\f90continuation.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\regression\\incfile.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\regression\\inout.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\regression\\lower_f2py_fortran.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\return_character\\foo77.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\return_character\\foo90.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\return_complex\\foo77.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\return_complex\\foo90.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\return_integer\\foo77.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\return_integer\\foo90.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\return_logical\\foo77.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\return_logical\\foo90.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\return_real\\foo77.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\return_real\\foo90.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\routines\\funcfortranname.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\routines\\funcfortranname.pyf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\routines\\subrout.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\routines\\subrout.pyf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\size\\foo.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\string\\char.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\string\\fixed_string.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\string\\gh24008.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\string\\gh24662.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\string\\gh25286.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\string\\gh25286.pyf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\string\\gh25286_bc.pyf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\string\\scalar_string.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\string\\string.f', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\src\\value_attrspec\\gh21665.f90', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_abstract_interface.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_array_from_pyobj.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_assumed_shape.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_block_docstring.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_callback.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_character.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_crackfortran.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_data.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_docs.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_f2cmap.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_f2py2e.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_isoc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_kind.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_mixed.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_modules.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_parameter.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_pyf_src.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_quoted_character.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_regression.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_return_character.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_return_complex.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_return_integer.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_return_logical.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_return_real.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_routines.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_semicolon_split.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_size.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_string.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_symbolic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\test_value_attrspec.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\tests\\util.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\f2py\\use_rules.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\fft\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\fft\\__init__.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\fft\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\fft\\__pycache__\\_helper.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\fft\\__pycache__\\_pocketfft.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\fft\\__pycache__\\helper.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\fft\\_helper.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\fft\\_helper.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\fft\\_pocketfft.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\fft\\_pocketfft.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\fft\\_pocketfft_umath.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\fft\\_pocketfft_umath.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\fft\\helper.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\fft\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\fft\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\fft\\tests\\__pycache__\\test_helper.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\fft\\tests\\__pycache__\\test_pocketfft.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\fft\\tests\\test_helper.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\fft\\tests\\test_pocketfft.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\__init__.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\_array_utils_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\_arraypad_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\_arraysetops_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\_arrayterator_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\_datasource.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\_function_base_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\_histograms_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\_index_tricks_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\_iotools.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\_nanfunctions_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\_npyio_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\_polynomial_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\_scimath_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\_shape_base_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\_stride_tricks_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\_twodim_base_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\_type_check_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\_ufunclike_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\_user_array_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\_utils_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\_version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\array_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\format.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\introspect.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\mixins.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\npyio.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\recfunctions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\scimath.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\stride_tricks.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\__pycache__\\user_array.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\_array_utils_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_array_utils_impl.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_arraypad_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_arraypad_impl.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_arraysetops_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_arraysetops_impl.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_arrayterator_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_arrayterator_impl.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_datasource.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_datasource.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_function_base_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_function_base_impl.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_histograms_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_histograms_impl.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_index_tricks_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_index_tricks_impl.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_iotools.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_iotools.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_nanfunctions_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_nanfunctions_impl.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_npyio_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_npyio_impl.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_polynomial_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_polynomial_impl.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_scimath_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_scimath_impl.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_shape_base_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_shape_base_impl.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_stride_tricks_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_stride_tricks_impl.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_twodim_base_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_twodim_base_impl.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_type_check_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_type_check_impl.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_ufunclike_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_ufunclike_impl.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_user_array_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_user_array_impl.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_utils_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_utils_impl.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\_version.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\array_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\array_utils.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\format.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\format.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\introspect.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\introspect.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\mixins.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\mixins.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\npyio.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\npyio.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\recfunctions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\recfunctions.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\scimath.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\scimath.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\stride_tricks.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\stride_tricks.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test__datasource.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test__iotools.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test__version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test_array_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test_arraypad.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test_arraysetops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test_arrayterator.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test_format.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test_function_base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test_histograms.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test_index_tricks.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test_io.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test_loadtxt.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test_mixins.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test_nanfunctions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test_packbits.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test_polynomial.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test_recfunctions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test_regression.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test_shape_base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test_stride_tricks.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test_twodim_base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test_type_check.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test_ufunclike.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\__pycache__\\test_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\data\\py2-np0-objarr.npy', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\data\\py2-objarr.npy', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\data\\py2-objarr.npz', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\data\\py3-objarr.npy', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\data\\py3-objarr.npz', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\data\\python3.npy', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\data\\win64python2.npy', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\lib\\tests\\test__datasource.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test__iotools.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test__version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test_array_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test_arraypad.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test_arraysetops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test_arrayterator.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test_format.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test_function_base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test_histograms.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test_index_tricks.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test_io.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test_loadtxt.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test_mixins.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test_nanfunctions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test_packbits.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test_polynomial.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test_recfunctions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test_regression.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test_shape_base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test_stride_tricks.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test_twodim_base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test_type_check.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test_ufunclike.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\tests\\test_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\user_array.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\lib\\user_array.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\linalg\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\linalg\\__init__.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\linalg\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\linalg\\__pycache__\\_linalg.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\linalg\\__pycache__\\linalg.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\linalg\\_linalg.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\linalg\\_linalg.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\linalg\\_umath_linalg.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\linalg\\_umath_linalg.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\linalg\\lapack_lite.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\linalg\\lapack_lite.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\linalg\\linalg.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\linalg\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\linalg\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\linalg\\tests\\__pycache__\\test_deprecations.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\linalg\\tests\\__pycache__\\test_linalg.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\linalg\\tests\\__pycache__\\test_regression.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\linalg\\tests\\test_deprecations.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\linalg\\tests\\test_linalg.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\linalg\\tests\\test_regression.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ma\\API_CHANGES.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ma\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ma\\README.rst', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ma\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ma\\__init__.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ma\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\ma\\__pycache__\\core.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\ma\\__pycache__\\extras.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\ma\\__pycache__\\mrecords.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\ma\\__pycache__\\testutils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\ma\\__pycache__\\timer_comparison.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\ma\\core.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ma\\core.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ma\\extras.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ma\\extras.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ma\\mrecords.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ma\\mrecords.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ma\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ma\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\ma\\tests\\__pycache__\\test_arrayobject.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\ma\\tests\\__pycache__\\test_core.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\ma\\tests\\__pycache__\\test_deprecations.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\ma\\tests\\__pycache__\\test_extras.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\ma\\tests\\__pycache__\\test_mrecords.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\ma\\tests\\__pycache__\\test_old_ma.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\ma\\tests\\__pycache__\\test_regression.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\ma\\tests\\__pycache__\\test_subclassing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\ma\\tests\\test_arrayobject.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ma\\tests\\test_core.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ma\\tests\\test_deprecations.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ma\\tests\\test_extras.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ma\\tests\\test_mrecords.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ma\\tests\\test_old_ma.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ma\\tests\\test_regression.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ma\\tests\\test_subclassing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ma\\testutils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\ma\\timer_comparison.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\matlib.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\matlib.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\matrixlib\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\matrixlib\\__init__.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\matrixlib\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\matrixlib\\__pycache__\\defmatrix.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\matrixlib\\defmatrix.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\matrixlib\\defmatrix.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\matrixlib\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\matrixlib\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\matrixlib\\tests\\__pycache__\\test_defmatrix.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\matrixlib\\tests\\__pycache__\\test_interaction.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\matrixlib\\tests\\__pycache__\\test_masked_matrix.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\matrixlib\\tests\\__pycache__\\test_matrix_linalg.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\matrixlib\\tests\\__pycache__\\test_multiarray.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\matrixlib\\tests\\__pycache__\\test_numeric.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\matrixlib\\tests\\__pycache__\\test_regression.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\matrixlib\\tests\\test_defmatrix.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\matrixlib\\tests\\test_interaction.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\matrixlib\\tests\\test_masked_matrix.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\matrixlib\\tests\\test_matrix_linalg.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\matrixlib\\tests\\test_multiarray.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\matrixlib\\tests\\test_numeric.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\matrixlib\\tests\\test_regression.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\__init__.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\polynomial\\__pycache__\\_polybase.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\polynomial\\__pycache__\\chebyshev.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\polynomial\\__pycache__\\hermite.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\polynomial\\__pycache__\\hermite_e.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\polynomial\\__pycache__\\laguerre.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\polynomial\\__pycache__\\legendre.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\polynomial\\__pycache__\\polynomial.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\polynomial\\__pycache__\\polyutils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\polynomial\\_polybase.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\_polybase.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\_polytypes.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\chebyshev.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\chebyshev.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\hermite.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\hermite.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\hermite_e.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\hermite_e.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\laguerre.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\laguerre.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\legendre.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\legendre.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\polynomial.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\polynomial.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\polyutils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\polyutils.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\polynomial\\tests\\__pycache__\\test_chebyshev.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\polynomial\\tests\\__pycache__\\test_classes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\polynomial\\tests\\__pycache__\\test_hermite.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\polynomial\\tests\\__pycache__\\test_hermite_e.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\polynomial\\tests\\__pycache__\\test_laguerre.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\polynomial\\tests\\__pycache__\\test_legendre.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\polynomial\\tests\\__pycache__\\test_polynomial.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\polynomial\\tests\\__pycache__\\test_polyutils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\polynomial\\tests\\__pycache__\\test_printing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\polynomial\\tests\\__pycache__\\test_symbol.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\polynomial\\tests\\test_chebyshev.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\tests\\test_classes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\tests\\test_hermite.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\tests\\test_hermite_e.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\tests\\test_laguerre.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\tests\\test_legendre.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\tests\\test_polynomial.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\tests\\test_polyutils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\tests\\test_printing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\polynomial\\tests\\test_symbol.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\LICENSE.md', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\__init__.pxd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\__init__.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\__pycache__\\_pickle.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\_bounded_integers.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\_bounded_integers.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\_bounded_integers.pxd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\_common.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\_common.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\_common.pxd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\_examples\\cffi\\__pycache__\\extending.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\_examples\\cffi\\__pycache__\\parse.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\_examples\\cffi\\extending.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\_examples\\cffi\\parse.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\_examples\\cython\\extending.pyx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\_examples\\cython\\extending_distributions.pyx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\_examples\\cython\\meson.build', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\_examples\\numba\\__pycache__\\extending.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\_examples\\numba\\__pycache__\\extending_distributions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\_examples\\numba\\extending.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\_examples\\numba\\extending_distributions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\_generator.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\_generator.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\_generator.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\_mt19937.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\_mt19937.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\_mt19937.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\_pcg64.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\_pcg64.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\_pcg64.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\_philox.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\_philox.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\_philox.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\_pickle.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\_sfc64.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\_sfc64.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\_sfc64.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\bit_generator.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\bit_generator.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\bit_generator.pxd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\bit_generator.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\c_distributions.pxd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\lib\\npyrandom.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\mtrand.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\mtrand.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\mtrand.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\tests\\__pycache__\\test_direct.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\tests\\__pycache__\\test_extending.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\tests\\__pycache__\\test_generator_mt19937.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\tests\\__pycache__\\test_generator_mt19937_regressions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\tests\\__pycache__\\test_random.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\tests\\__pycache__\\test_randomstate.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\tests\\__pycache__\\test_randomstate_regression.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\tests\\__pycache__\\test_regression.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\tests\\__pycache__\\test_seed_sequence.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\tests\\__pycache__\\test_smoke.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\tests\\data\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\tests\\data\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\tests\\data\\generator_pcg64_np121.pkl.gz', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\tests\\data\\generator_pcg64_np126.pkl.gz', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\tests\\data\\mt19937-testset-1.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\tests\\data\\mt19937-testset-2.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\tests\\data\\pcg64-testset-1.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\tests\\data\\pcg64-testset-2.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\tests\\data\\pcg64dxsm-testset-1.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\tests\\data\\pcg64dxsm-testset-2.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\tests\\data\\philox-testset-1.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\tests\\data\\philox-testset-2.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\tests\\data\\sfc64-testset-1.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\tests\\data\\sfc64-testset-2.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\tests\\data\\sfc64_np126.pkl.gz', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\random\\tests\\test_direct.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\tests\\test_extending.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\tests\\test_generator_mt19937.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\tests\\test_generator_mt19937_regressions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\tests\\test_random.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\tests\\test_randomstate.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\tests\\test_randomstate_regression.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\tests\\test_regression.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\tests\\test_seed_sequence.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\random\\tests\\test_smoke.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\rec\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\rec\\__init__.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\rec\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\strings\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\strings\\__init__.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\strings\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\testing\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\testing\\__init__.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\testing\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\testing\\__pycache__\\overrides.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\testing\\__pycache__\\print_coercion_tables.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\testing\\_private\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\testing\\_private\\__init__.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\testing\\_private\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\testing\\_private\\__pycache__\\extbuild.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\testing\\_private\\__pycache__\\utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\testing\\_private\\extbuild.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\testing\\_private\\extbuild.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\testing\\_private\\utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\testing\\_private\\utils.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\testing\\overrides.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\testing\\overrides.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\testing\\print_coercion_tables.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\testing\\print_coercion_tables.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\testing\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\testing\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\testing\\tests\\__pycache__\\test_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\testing\\tests\\test_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\tests\\__pycache__\\test__all__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\tests\\__pycache__\\test_configtool.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\tests\\__pycache__\\test_ctypeslib.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\tests\\__pycache__\\test_lazyloading.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\tests\\__pycache__\\test_matlib.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\tests\\__pycache__\\test_numpy_config.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\tests\\__pycache__\\test_numpy_version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\tests\\__pycache__\\test_public_api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\tests\\__pycache__\\test_reloading.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\tests\\__pycache__\\test_scripts.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\tests\\__pycache__\\test_warnings.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\tests\\test__all__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\tests\\test_configtool.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\tests\\test_ctypeslib.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\tests\\test_lazyloading.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\tests\\test_matlib.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\tests\\test_numpy_config.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\tests\\test_numpy_version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\tests\\test_public_api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\tests\\test_reloading.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\tests\\test_scripts.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\tests\\test_warnings.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\__pycache__\\mypy_plugin.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\mypy_plugin.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\__pycache__\\test_isfile.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\__pycache__\\test_runtime.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\__pycache__\\test_typing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\arithmetic.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\array_constructors.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\array_like.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\array_pad.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\arrayprint.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\arrayterator.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\bitwise_ops.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\char.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\chararray.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\comparisons.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\constants.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\datasource.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\dtype.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\einsumfunc.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\flatiter.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\fromnumeric.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\histograms.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\index_tricks.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\lib_function_base.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\lib_polynomial.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\lib_utils.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\lib_version.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\linalg.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\memmap.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\modules.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\multiarray.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\ndarray.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\ndarray_misc.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\nditer.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\nested_sequence.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\npyio.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\numerictypes.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\random.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\rec.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\scalars.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\shape.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\shape_base.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\stride_tricks.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\strings.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\testing.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\twodim_base.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\type_check.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\ufunc_config.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\ufunclike.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\ufuncs.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\fail\\warnings_and_errors.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\misc\\extended_precision.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\mypy.ini', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\arithmetic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\array_constructors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\array_like.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\arrayprint.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\arrayterator.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\bitwise_ops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\comparisons.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\dtype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\einsumfunc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\flatiter.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\fromnumeric.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\index_tricks.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\lib_user_array.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\lib_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\lib_version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\literal.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\ma.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\mod.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\modules.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\multiarray.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\ndarray_conversion.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\ndarray_misc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\ndarray_shape_manipulation.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\nditer.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\numeric.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\numerictypes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\random.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\recfunctions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\scalars.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\shape.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\simple.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\simple_py3.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\ufunc_config.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\ufunclike.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\ufuncs.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\__pycache__\\warnings_and_errors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\arithmetic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\array_constructors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\array_like.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\arrayprint.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\arrayterator.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\bitwise_ops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\comparisons.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\dtype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\einsumfunc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\flatiter.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\fromnumeric.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\index_tricks.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\lib_user_array.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\lib_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\lib_version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\literal.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\ma.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\mod.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\modules.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\multiarray.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\ndarray_conversion.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\ndarray_misc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\ndarray_shape_manipulation.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\nditer.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\numeric.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\numerictypes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\random.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\recfunctions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\scalars.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\shape.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\simple.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\simple_py3.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\ufunc_config.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\ufunclike.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\ufuncs.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\pass\\warnings_and_errors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\arithmetic.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\array_api_info.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\array_constructors.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\arraypad.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\arrayprint.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\arraysetops.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\arrayterator.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\bitwise_ops.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\char.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\chararray.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\comparisons.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\constants.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\ctypeslib.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\datasource.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\dtype.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\einsumfunc.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\emath.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\fft.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\flatiter.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\fromnumeric.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\getlimits.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\histograms.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\index_tricks.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\lib_function_base.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\lib_polynomial.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\lib_utils.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\lib_version.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\linalg.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\matrix.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\memmap.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\mod.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\modules.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\multiarray.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\nbit_base_example.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\ndarray_assignability.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\ndarray_conversion.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\ndarray_misc.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\ndarray_shape_manipulation.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\nditer.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\nested_sequence.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\npyio.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\numeric.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\numerictypes.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\polynomial_polybase.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\polynomial_polyutils.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\polynomial_series.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\random.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\rec.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\scalars.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\shape.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\shape_base.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\stride_tricks.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\strings.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\testing.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\twodim_base.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\type_check.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\ufunc_config.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\ufunclike.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\ufuncs.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\data\\reveal\\warnings_and_errors.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\test_isfile.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\test_runtime.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\typing\\tests\\test_typing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\numpy\\version.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging-24.2.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging-24.2.dist-info\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging-24.2.dist-info\\LICENSE.APACHE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging-24.2.dist-info\\LICENSE.BSD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging-24.2.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging-24.2.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging-24.2.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\_elffile.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\_manylinux.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\_musllinux.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\_parser.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\_structures.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\_tokenizer.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\markers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\metadata.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\requirements.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\specifiers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\tags.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\packaging\\__pycache__\\version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\packaging\\_elffile.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging\\_manylinux.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging\\_musllinux.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging\\_parser.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging\\_structures.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging\\_tokenizer.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging\\licenses\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging\\licenses\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\packaging\\licenses\\__pycache__\\_spdx.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\packaging\\licenses\\_spdx.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging\\markers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging\\metadata.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging\\requirements.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging\\specifiers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging\\tags.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging\\utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\packaging\\version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas-2.2.3.dist-info\\DELVEWHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas-2.2.3.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas-2.2.3.dist-info\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas-2.2.3.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas-2.2.3.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas-2.2.3.dist-info\\REQUESTED', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas-2.2.3.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas-2.2.3.dist-info\\entry_points.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas.libs\\msvcp140-0f2ea95580b32bcfc81c235d5751ce78.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\__pycache__\\_typing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\__pycache__\\_version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\__pycache__\\_version_meson.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\__pycache__\\testing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_config\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_config\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_config\\__pycache__\\config.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_config\\__pycache__\\dates.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_config\\__pycache__\\display.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_config\\__pycache__\\localization.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_config\\config.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_config\\dates.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_config\\display.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_config\\localization.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\algos.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\algos.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\algos.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\arrays.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\arrays.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\arrays.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\byteswap.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\byteswap.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\byteswap.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\groupby.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\groupby.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\groupby.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\hashing.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\hashing.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\hashing.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\hashtable.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\hashtable.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\hashtable.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\index.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\index.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\index.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\indexing.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\indexing.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\indexing.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\internals.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\internals.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\internals.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\interval.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\interval.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\interval.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\join.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\join.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\join.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\json.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\json.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\json.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\lib.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\lib.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\lib.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\missing.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\missing.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\missing.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\ops.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\ops.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\ops.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\ops_dispatch.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\ops_dispatch.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\ops_dispatch.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\pandas_datetime.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\pandas_datetime.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\pandas_parser.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\pandas_parser.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\parsers.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\parsers.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\parsers.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\properties.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\properties.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\properties.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\reshape.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\reshape.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\reshape.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\sas.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\sas.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\sas.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\sparse.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\sparse.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\sparse.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\testing.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\testing.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\testing.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\tslib.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslib.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslib.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\base.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\base.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\ccalendar.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\ccalendar.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\ccalendar.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\conversion.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\conversion.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\conversion.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\dtypes.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\dtypes.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\dtypes.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\fields.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\fields.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\fields.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\nattype.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\nattype.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\nattype.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\np_datetime.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\np_datetime.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\np_datetime.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\offsets.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\offsets.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\offsets.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\parsing.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\parsing.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\parsing.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\period.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\period.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\period.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\strptime.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\strptime.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\strptime.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\timedeltas.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\timedeltas.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\timedeltas.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\timestamps.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\timestamps.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\timestamps.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\timezones.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\timezones.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\timezones.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\tzconversion.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\tzconversion.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\tzconversion.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\vectorized.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\vectorized.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\tslibs\\vectorized.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\window\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\window\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\window\\aggregations.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\window\\aggregations.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\window\\aggregations.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\window\\indexers.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\window\\indexers.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\window\\indexers.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_libs\\writers.cp310-win_amd64.lib', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\writers.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_libs\\writers.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_testing\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_testing\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_testing\\__pycache__\\_hypothesis.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_testing\\__pycache__\\_io.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_testing\\__pycache__\\_warnings.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_testing\\__pycache__\\asserters.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_testing\\__pycache__\\compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_testing\\__pycache__\\contexts.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\_testing\\_hypothesis.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_testing\\_io.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_testing\\_warnings.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_testing\\asserters.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_testing\\compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_testing\\contexts.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_typing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\_version_meson.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\api\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\api\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\api\\extensions\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\api\\extensions\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\api\\indexers\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\api\\indexers\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\api\\interchange\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\api\\interchange\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\api\\types\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\api\\types\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\api\\typing\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\api\\typing\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\arrays\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\arrays\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\compat\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\compat\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\compat\\__pycache__\\_constants.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\compat\\__pycache__\\_optional.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\compat\\__pycache__\\compressors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\compat\\__pycache__\\pickle_compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\compat\\__pycache__\\pyarrow.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\compat\\_constants.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\compat\\_optional.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\compat\\compressors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\compat\\numpy\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\compat\\numpy\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\compat\\numpy\\__pycache__\\function.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\compat\\numpy\\function.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\compat\\pickle_compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\compat\\pyarrow.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\__pycache__\\accessor.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\__pycache__\\algorithms.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\__pycache__\\api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\__pycache__\\apply.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\__pycache__\\arraylike.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\__pycache__\\base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\__pycache__\\common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\__pycache__\\config_init.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\__pycache__\\construction.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\__pycache__\\flags.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\__pycache__\\frame.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\__pycache__\\generic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\__pycache__\\indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\__pycache__\\missing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\__pycache__\\nanops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\__pycache__\\resample.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\__pycache__\\roperator.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\__pycache__\\sample.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\__pycache__\\series.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\__pycache__\\shared_docs.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\__pycache__\\sorting.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\_numba\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\_numba\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\_numba\\__pycache__\\executor.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\_numba\\__pycache__\\extensions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\_numba\\executor.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\_numba\\extensions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\_numba\\kernels\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\_numba\\kernels\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\_numba\\kernels\\__pycache__\\mean_.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\_numba\\kernels\\__pycache__\\min_max_.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\_numba\\kernels\\__pycache__\\shared.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\_numba\\kernels\\__pycache__\\sum_.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\_numba\\kernels\\__pycache__\\var_.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\_numba\\kernels\\mean_.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\_numba\\kernels\\min_max_.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\_numba\\kernels\\shared.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\_numba\\kernels\\sum_.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\_numba\\kernels\\var_.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\accessor.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\algorithms.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\apply.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\array_algos\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\array_algos\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\array_algos\\__pycache__\\datetimelike_accumulations.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\array_algos\\__pycache__\\masked_accumulations.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\array_algos\\__pycache__\\masked_reductions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\array_algos\\__pycache__\\putmask.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\array_algos\\__pycache__\\quantile.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\array_algos\\__pycache__\\replace.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\array_algos\\__pycache__\\take.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\array_algos\\__pycache__\\transforms.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\array_algos\\datetimelike_accumulations.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\array_algos\\masked_accumulations.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\array_algos\\masked_reductions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\array_algos\\putmask.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\array_algos\\quantile.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\array_algos\\replace.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\array_algos\\take.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\array_algos\\transforms.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arraylike.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\__pycache__\\_arrow_string_mixins.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\__pycache__\\_mixins.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\__pycache__\\_ranges.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\__pycache__\\_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\__pycache__\\base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\__pycache__\\boolean.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\__pycache__\\categorical.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\__pycache__\\datetimelike.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\__pycache__\\datetimes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\__pycache__\\floating.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\__pycache__\\integer.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\__pycache__\\interval.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\__pycache__\\masked.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\__pycache__\\numeric.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\__pycache__\\numpy_.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\__pycache__\\period.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\__pycache__\\string_.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\__pycache__\\string_arrow.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\__pycache__\\timedeltas.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\_arrow_string_mixins.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\_mixins.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\_ranges.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\arrow\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\arrow\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\arrow\\__pycache__\\_arrow_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\arrow\\__pycache__\\accessors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\arrow\\__pycache__\\array.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\arrow\\__pycache__\\extension_types.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\arrow\\_arrow_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\arrow\\accessors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\arrow\\array.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\arrow\\extension_types.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\boolean.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\categorical.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\datetimelike.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\datetimes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\floating.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\integer.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\interval.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\masked.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\numeric.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\numpy_.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\period.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\sparse\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\sparse\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\sparse\\__pycache__\\accessor.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\sparse\\__pycache__\\array.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\sparse\\__pycache__\\scipy_sparse.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\arrays\\sparse\\accessor.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\sparse\\array.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\sparse\\scipy_sparse.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\string_.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\string_arrow.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\arrays\\timedeltas.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\computation\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\computation\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\computation\\__pycache__\\align.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\computation\\__pycache__\\api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\computation\\__pycache__\\check.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\computation\\__pycache__\\common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\computation\\__pycache__\\engines.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\computation\\__pycache__\\eval.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\computation\\__pycache__\\expr.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\computation\\__pycache__\\expressions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\computation\\__pycache__\\ops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\computation\\__pycache__\\parsing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\computation\\__pycache__\\pytables.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\computation\\__pycache__\\scope.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\computation\\align.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\computation\\api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\computation\\check.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\computation\\common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\computation\\engines.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\computation\\eval.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\computation\\expr.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\computation\\expressions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\computation\\ops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\computation\\parsing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\computation\\pytables.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\computation\\scope.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\config_init.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\construction.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\dtypes\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\dtypes\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\dtypes\\__pycache__\\api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\dtypes\\__pycache__\\astype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\dtypes\\__pycache__\\base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\dtypes\\__pycache__\\cast.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\dtypes\\__pycache__\\common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\dtypes\\__pycache__\\concat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\dtypes\\__pycache__\\dtypes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\dtypes\\__pycache__\\generic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\dtypes\\__pycache__\\inference.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\dtypes\\__pycache__\\missing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\dtypes\\api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\dtypes\\astype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\dtypes\\base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\dtypes\\cast.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\dtypes\\common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\dtypes\\concat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\dtypes\\dtypes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\dtypes\\generic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\dtypes\\inference.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\dtypes\\missing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\flags.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\frame.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\generic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\groupby\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\groupby\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\groupby\\__pycache__\\base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\groupby\\__pycache__\\categorical.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\groupby\\__pycache__\\generic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\groupby\\__pycache__\\groupby.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\groupby\\__pycache__\\grouper.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\groupby\\__pycache__\\indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\groupby\\__pycache__\\numba_.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\groupby\\__pycache__\\ops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\groupby\\base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\groupby\\categorical.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\groupby\\generic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\groupby\\groupby.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\groupby\\grouper.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\groupby\\indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\groupby\\numba_.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\groupby\\ops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\indexers\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\indexers\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\indexers\\__pycache__\\objects.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\indexers\\__pycache__\\utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\indexers\\objects.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\indexers\\utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\indexes\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\indexes\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\indexes\\__pycache__\\accessors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\indexes\\__pycache__\\api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\indexes\\__pycache__\\base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\indexes\\__pycache__\\category.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\indexes\\__pycache__\\datetimelike.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\indexes\\__pycache__\\datetimes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\indexes\\__pycache__\\extension.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\indexes\\__pycache__\\frozen.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\indexes\\__pycache__\\interval.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\indexes\\__pycache__\\multi.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\indexes\\__pycache__\\period.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\indexes\\__pycache__\\range.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\indexes\\__pycache__\\timedeltas.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\indexes\\accessors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\indexes\\api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\indexes\\base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\indexes\\category.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\indexes\\datetimelike.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\indexes\\datetimes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\indexes\\extension.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\indexes\\frozen.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\indexes\\interval.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\indexes\\multi.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\indexes\\period.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\indexes\\range.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\indexes\\timedeltas.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\interchange\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\interchange\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\interchange\\__pycache__\\buffer.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\interchange\\__pycache__\\column.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\interchange\\__pycache__\\dataframe.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\interchange\\__pycache__\\dataframe_protocol.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\interchange\\__pycache__\\from_dataframe.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\interchange\\__pycache__\\utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\interchange\\buffer.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\interchange\\column.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\interchange\\dataframe.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\interchange\\dataframe_protocol.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\interchange\\from_dataframe.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\interchange\\utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\internals\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\internals\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\internals\\__pycache__\\api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\internals\\__pycache__\\array_manager.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\internals\\__pycache__\\base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\internals\\__pycache__\\blocks.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\internals\\__pycache__\\concat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\internals\\__pycache__\\construction.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\internals\\__pycache__\\managers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\internals\\__pycache__\\ops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\internals\\api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\internals\\array_manager.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\internals\\base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\internals\\blocks.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\internals\\concat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\internals\\construction.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\internals\\managers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\internals\\ops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\methods\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\methods\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\methods\\__pycache__\\describe.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\methods\\__pycache__\\selectn.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\methods\\__pycache__\\to_dict.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\methods\\describe.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\methods\\selectn.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\methods\\to_dict.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\missing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\nanops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\ops\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\ops\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\ops\\__pycache__\\array_ops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\ops\\__pycache__\\common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\ops\\__pycache__\\dispatch.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\ops\\__pycache__\\docstrings.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\ops\\__pycache__\\invalid.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\ops\\__pycache__\\mask_ops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\ops\\__pycache__\\missing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\ops\\array_ops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\ops\\common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\ops\\dispatch.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\ops\\docstrings.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\ops\\invalid.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\ops\\mask_ops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\ops\\missing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\resample.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\reshape\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\reshape\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\reshape\\__pycache__\\api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\reshape\\__pycache__\\concat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\reshape\\__pycache__\\encoding.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\reshape\\__pycache__\\melt.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\reshape\\__pycache__\\merge.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\reshape\\__pycache__\\pivot.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\reshape\\__pycache__\\reshape.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\reshape\\__pycache__\\tile.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\reshape\\__pycache__\\util.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\reshape\\api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\reshape\\concat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\reshape\\encoding.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\reshape\\melt.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\reshape\\merge.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\reshape\\pivot.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\reshape\\reshape.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\reshape\\tile.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\reshape\\util.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\roperator.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\sample.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\series.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\shared_docs.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\sorting.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\sparse\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\sparse\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\sparse\\__pycache__\\api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\sparse\\api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\strings\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\strings\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\strings\\__pycache__\\accessor.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\strings\\__pycache__\\base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\strings\\__pycache__\\object_array.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\strings\\accessor.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\strings\\base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\strings\\object_array.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\tools\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\tools\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\tools\\__pycache__\\datetimes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\tools\\__pycache__\\numeric.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\tools\\__pycache__\\timedeltas.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\tools\\__pycache__\\times.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\tools\\datetimes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\tools\\numeric.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\tools\\timedeltas.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\tools\\times.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\util\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\util\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\util\\__pycache__\\hashing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\util\\__pycache__\\numba_.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\util\\hashing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\util\\numba_.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\window\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\window\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\window\\__pycache__\\common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\window\\__pycache__\\doc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\window\\__pycache__\\ewm.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\window\\__pycache__\\expanding.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\window\\__pycache__\\numba_.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\window\\__pycache__\\online.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\window\\__pycache__\\rolling.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\core\\window\\common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\window\\doc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\window\\ewm.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\window\\expanding.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\window\\numba_.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\window\\online.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\core\\window\\rolling.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\errors\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\errors\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\__pycache__\\_util.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\__pycache__\\api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\__pycache__\\clipboards.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\__pycache__\\common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\__pycache__\\feather_format.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\__pycache__\\gbq.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\__pycache__\\html.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\__pycache__\\orc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\__pycache__\\parquet.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\__pycache__\\pickle.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\__pycache__\\pytables.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\__pycache__\\spss.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\__pycache__\\sql.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\__pycache__\\stata.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\__pycache__\\xml.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\_util.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\clipboard\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\clipboard\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\clipboards.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\excel\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\excel\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\excel\\__pycache__\\_base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\excel\\__pycache__\\_calamine.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\excel\\__pycache__\\_odfreader.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\excel\\__pycache__\\_odswriter.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\excel\\__pycache__\\_openpyxl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\excel\\__pycache__\\_pyxlsb.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\excel\\__pycache__\\_util.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\excel\\__pycache__\\_xlrd.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\excel\\__pycache__\\_xlsxwriter.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\excel\\_base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\excel\\_calamine.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\excel\\_odfreader.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\excel\\_odswriter.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\excel\\_openpyxl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\excel\\_pyxlsb.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\excel\\_util.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\excel\\_xlrd.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\excel\\_xlsxwriter.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\feather_format.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\formats\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\formats\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\formats\\__pycache__\\_color_data.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\formats\\__pycache__\\console.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\formats\\__pycache__\\css.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\formats\\__pycache__\\csvs.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\formats\\__pycache__\\excel.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\formats\\__pycache__\\format.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\formats\\__pycache__\\html.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\formats\\__pycache__\\info.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\formats\\__pycache__\\printing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\formats\\__pycache__\\string.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\formats\\__pycache__\\style.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\formats\\__pycache__\\style_render.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\formats\\__pycache__\\xml.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\formats\\_color_data.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\formats\\console.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\formats\\css.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\formats\\csvs.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\formats\\excel.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\formats\\format.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\formats\\html.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\formats\\info.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\formats\\printing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\formats\\string.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\formats\\style.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\formats\\style_render.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\formats\\templates\\html.tpl', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\formats\\templates\\html_style.tpl', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\formats\\templates\\html_table.tpl', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\formats\\templates\\latex.tpl', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\formats\\templates\\latex_longtable.tpl', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\formats\\templates\\latex_table.tpl', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\formats\\templates\\string.tpl', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\formats\\xml.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\gbq.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\html.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\json\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\json\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\json\\__pycache__\\_json.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\json\\__pycache__\\_normalize.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\json\\__pycache__\\_table_schema.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\json\\_json.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\json\\_normalize.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\json\\_table_schema.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\orc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\parquet.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\parsers\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\parsers\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\parsers\\__pycache__\\arrow_parser_wrapper.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\parsers\\__pycache__\\base_parser.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\parsers\\__pycache__\\c_parser_wrapper.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\parsers\\__pycache__\\python_parser.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\parsers\\__pycache__\\readers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\parsers\\arrow_parser_wrapper.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\parsers\\base_parser.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\parsers\\c_parser_wrapper.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\parsers\\python_parser.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\parsers\\readers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\pickle.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\pytables.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\sas\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\sas\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\sas\\__pycache__\\sas7bdat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\sas\\__pycache__\\sas_constants.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\sas\\__pycache__\\sas_xport.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\sas\\__pycache__\\sasreader.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\io\\sas\\sas7bdat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\sas\\sas_constants.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\sas\\sas_xport.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\sas\\sasreader.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\spss.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\sql.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\stata.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\io\\xml.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\plotting\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\plotting\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\plotting\\__pycache__\\_core.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\plotting\\__pycache__\\_misc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\plotting\\_core.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\plotting\\_matplotlib\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\plotting\\_matplotlib\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\plotting\\_matplotlib\\__pycache__\\boxplot.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\plotting\\_matplotlib\\__pycache__\\converter.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\plotting\\_matplotlib\\__pycache__\\core.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\plotting\\_matplotlib\\__pycache__\\groupby.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\plotting\\_matplotlib\\__pycache__\\hist.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\plotting\\_matplotlib\\__pycache__\\misc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\plotting\\_matplotlib\\__pycache__\\style.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\plotting\\_matplotlib\\__pycache__\\timeseries.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\plotting\\_matplotlib\\__pycache__\\tools.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\plotting\\_matplotlib\\boxplot.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\plotting\\_matplotlib\\converter.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\plotting\\_matplotlib\\core.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\plotting\\_matplotlib\\groupby.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\plotting\\_matplotlib\\hist.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\plotting\\_matplotlib\\misc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\plotting\\_matplotlib\\style.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\plotting\\_matplotlib\\timeseries.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\plotting\\_matplotlib\\tools.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\plotting\\_misc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\pyproject.toml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\testing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\__pycache__\\test_aggregation.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\__pycache__\\test_algos.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\__pycache__\\test_common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\__pycache__\\test_downstream.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\__pycache__\\test_errors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\__pycache__\\test_expressions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\__pycache__\\test_flags.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\__pycache__\\test_multilevel.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\__pycache__\\test_nanops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\__pycache__\\test_optional_dependency.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\__pycache__\\test_register_accessor.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\__pycache__\\test_sorting.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\__pycache__\\test_take.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\api\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\api\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\api\\__pycache__\\test_api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\api\\__pycache__\\test_types.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\api\\test_api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\api\\test_types.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\apply\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\apply\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\apply\\__pycache__\\common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\apply\\__pycache__\\test_frame_apply.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\apply\\__pycache__\\test_frame_apply_relabeling.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\apply\\__pycache__\\test_frame_transform.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\apply\\__pycache__\\test_invalid_arg.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\apply\\__pycache__\\test_numba.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\apply\\__pycache__\\test_series_apply.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\apply\\__pycache__\\test_series_apply_relabeling.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\apply\\__pycache__\\test_series_transform.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\apply\\__pycache__\\test_str.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\apply\\common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\apply\\test_frame_apply.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\apply\\test_frame_apply_relabeling.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\apply\\test_frame_transform.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\apply\\test_invalid_arg.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\apply\\test_numba.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\apply\\test_series_apply.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\apply\\test_series_apply_relabeling.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\apply\\test_series_transform.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\apply\\test_str.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arithmetic\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arithmetic\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arithmetic\\__pycache__\\common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arithmetic\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arithmetic\\__pycache__\\test_array_ops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arithmetic\\__pycache__\\test_categorical.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arithmetic\\__pycache__\\test_datetime64.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arithmetic\\__pycache__\\test_interval.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arithmetic\\__pycache__\\test_numeric.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arithmetic\\__pycache__\\test_object.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arithmetic\\__pycache__\\test_period.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arithmetic\\__pycache__\\test_timedelta64.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arithmetic\\common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arithmetic\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arithmetic\\test_array_ops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arithmetic\\test_categorical.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arithmetic\\test_datetime64.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arithmetic\\test_interval.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arithmetic\\test_numeric.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arithmetic\\test_object.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arithmetic\\test_period.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arithmetic\\test_timedelta64.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\__pycache__\\masked_shared.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\__pycache__\\test_array.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\__pycache__\\test_datetimelike.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\__pycache__\\test_datetimes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\__pycache__\\test_ndarray_backed.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\__pycache__\\test_period.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\__pycache__\\test_timedeltas.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\boolean\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\boolean\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\boolean\\__pycache__\\test_arithmetic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\boolean\\__pycache__\\test_astype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\boolean\\__pycache__\\test_comparison.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\boolean\\__pycache__\\test_construction.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\boolean\\__pycache__\\test_function.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\boolean\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\boolean\\__pycache__\\test_logical.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\boolean\\__pycache__\\test_ops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\boolean\\__pycache__\\test_reduction.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\boolean\\__pycache__\\test_repr.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\boolean\\test_arithmetic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\boolean\\test_astype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\boolean\\test_comparison.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\boolean\\test_construction.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\boolean\\test_function.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\boolean\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\boolean\\test_logical.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\boolean\\test_ops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\boolean\\test_reduction.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\boolean\\test_repr.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\__pycache__\\test_algos.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\__pycache__\\test_analytics.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\__pycache__\\test_api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\__pycache__\\test_astype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\__pycache__\\test_constructors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\__pycache__\\test_dtypes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\__pycache__\\test_map.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\__pycache__\\test_missing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\__pycache__\\test_operators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\__pycache__\\test_replace.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\__pycache__\\test_repr.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\__pycache__\\test_sorting.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\__pycache__\\test_subclass.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\__pycache__\\test_take.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\__pycache__\\test_warnings.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\test_algos.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\test_analytics.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\test_api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\test_astype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\test_constructors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\test_dtypes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\test_map.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\test_missing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\test_operators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\test_replace.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\test_repr.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\test_sorting.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\test_subclass.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\test_take.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\categorical\\test_warnings.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\datetimes\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\datetimes\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\datetimes\\__pycache__\\test_constructors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\datetimes\\__pycache__\\test_cumulative.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\datetimes\\__pycache__\\test_reductions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\datetimes\\test_constructors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\datetimes\\test_cumulative.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\datetimes\\test_reductions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\floating\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\floating\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\floating\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\floating\\__pycache__\\test_arithmetic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\floating\\__pycache__\\test_astype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\floating\\__pycache__\\test_comparison.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\floating\\__pycache__\\test_concat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\floating\\__pycache__\\test_construction.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\floating\\__pycache__\\test_contains.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\floating\\__pycache__\\test_function.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\floating\\__pycache__\\test_repr.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\floating\\__pycache__\\test_to_numpy.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\floating\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\floating\\test_arithmetic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\floating\\test_astype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\floating\\test_comparison.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\floating\\test_concat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\floating\\test_construction.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\floating\\test_contains.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\floating\\test_function.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\floating\\test_repr.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\floating\\test_to_numpy.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\integer\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\integer\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\integer\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\integer\\__pycache__\\test_arithmetic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\integer\\__pycache__\\test_comparison.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\integer\\__pycache__\\test_concat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\integer\\__pycache__\\test_construction.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\integer\\__pycache__\\test_dtypes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\integer\\__pycache__\\test_function.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\integer\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\integer\\__pycache__\\test_reduction.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\integer\\__pycache__\\test_repr.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\integer\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\integer\\test_arithmetic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\integer\\test_comparison.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\integer\\test_concat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\integer\\test_construction.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\integer\\test_dtypes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\integer\\test_function.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\integer\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\integer\\test_reduction.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\integer\\test_repr.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\interval\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\interval\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\interval\\__pycache__\\test_astype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\interval\\__pycache__\\test_formats.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\interval\\__pycache__\\test_interval.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\interval\\__pycache__\\test_interval_pyarrow.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\interval\\__pycache__\\test_overlaps.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\interval\\test_astype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\interval\\test_formats.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\interval\\test_interval.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\interval\\test_interval_pyarrow.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\interval\\test_overlaps.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\masked\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\masked\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\masked\\__pycache__\\test_arithmetic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\masked\\__pycache__\\test_arrow_compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\masked\\__pycache__\\test_function.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\masked\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\masked\\test_arithmetic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\masked\\test_arrow_compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\masked\\test_function.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\masked\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\masked_shared.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\numpy_\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\numpy_\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\numpy_\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\numpy_\\__pycache__\\test_numpy.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\numpy_\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\numpy_\\test_numpy.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\period\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\period\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\period\\__pycache__\\test_arrow_compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\period\\__pycache__\\test_astype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\period\\__pycache__\\test_constructors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\period\\__pycache__\\test_reductions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\period\\test_arrow_compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\period\\test_astype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\period\\test_constructors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\period\\test_reductions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\__pycache__\\test_accessor.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\__pycache__\\test_arithmetics.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\__pycache__\\test_array.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\__pycache__\\test_astype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\__pycache__\\test_combine_concat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\__pycache__\\test_constructors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\__pycache__\\test_dtype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\__pycache__\\test_libsparse.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\__pycache__\\test_reductions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\__pycache__\\test_unary.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\test_accessor.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\test_arithmetics.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\test_array.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\test_astype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\test_combine_concat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\test_constructors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\test_dtype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\test_libsparse.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\test_reductions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\sparse\\test_unary.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\string_\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\string_\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\string_\\__pycache__\\test_string.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\string_\\__pycache__\\test_string_arrow.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\string_\\test_string.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\string_\\test_string_arrow.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\test_array.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\test_datetimelike.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\test_datetimes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\test_ndarray_backed.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\test_period.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\test_timedeltas.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\timedeltas\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\timedeltas\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\timedeltas\\__pycache__\\test_constructors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\timedeltas\\__pycache__\\test_cumulative.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\timedeltas\\__pycache__\\test_reductions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\arrays\\timedeltas\\test_constructors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\timedeltas\\test_cumulative.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\arrays\\timedeltas\\test_reductions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\base\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\base\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\base\\__pycache__\\common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\base\\__pycache__\\test_constructors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\base\\__pycache__\\test_conversion.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\base\\__pycache__\\test_fillna.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\base\\__pycache__\\test_misc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\base\\__pycache__\\test_transpose.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\base\\__pycache__\\test_unique.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\base\\__pycache__\\test_value_counts.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\base\\common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\base\\test_constructors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\base\\test_conversion.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\base\\test_fillna.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\base\\test_misc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\base\\test_transpose.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\base\\test_unique.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\base\\test_value_counts.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\computation\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\computation\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\computation\\__pycache__\\test_compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\computation\\__pycache__\\test_eval.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\computation\\test_compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\computation\\test_eval.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\config\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\config\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\config\\__pycache__\\test_config.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\config\\__pycache__\\test_localization.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\config\\test_config.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\config\\test_localization.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\construction\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\construction\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\construction\\__pycache__\\test_extract_array.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\construction\\test_extract_array.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\__pycache__\\test_array.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\__pycache__\\test_astype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\__pycache__\\test_chained_assignment_deprecation.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\__pycache__\\test_clip.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\__pycache__\\test_constructors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\__pycache__\\test_core_functionalities.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\__pycache__\\test_functions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\__pycache__\\test_internals.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\__pycache__\\test_interp_fillna.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\__pycache__\\test_methods.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\__pycache__\\test_replace.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\__pycache__\\test_setitem.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\__pycache__\\test_util.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\__pycache__\\util.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\index\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\index\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\index\\__pycache__\\test_datetimeindex.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\index\\__pycache__\\test_index.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\index\\__pycache__\\test_periodindex.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\index\\__pycache__\\test_timedeltaindex.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\index\\test_datetimeindex.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\index\\test_index.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\index\\test_periodindex.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\index\\test_timedeltaindex.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\test_array.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\test_astype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\test_chained_assignment_deprecation.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\test_clip.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\test_constructors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\test_core_functionalities.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\test_functions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\test_internals.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\test_interp_fillna.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\test_methods.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\test_replace.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\test_setitem.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\test_util.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\copy_view\\util.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\__pycache__\\test_common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\__pycache__\\test_concat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\__pycache__\\test_dtypes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\__pycache__\\test_generic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\__pycache__\\test_inference.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\__pycache__\\test_missing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\__pycache__\\test_can_hold_element.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\__pycache__\\test_construct_from_scalar.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\__pycache__\\test_construct_ndarray.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\__pycache__\\test_construct_object_arr.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\__pycache__\\test_dict_compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\__pycache__\\test_downcast.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\__pycache__\\test_find_common_type.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\__pycache__\\test_infer_datetimelike.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\__pycache__\\test_infer_dtype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\__pycache__\\test_maybe_box_native.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\__pycache__\\test_promote.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\test_can_hold_element.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\test_construct_from_scalar.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\test_construct_ndarray.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\test_construct_object_arr.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\test_dict_compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\test_downcast.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\test_find_common_type.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\test_infer_datetimelike.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\test_infer_dtype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\test_maybe_box_native.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\cast\\test_promote.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\test_common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\test_concat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\test_dtypes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\test_generic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\test_inference.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\dtypes\\test_missing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\__pycache__\\test_arrow.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\__pycache__\\test_categorical.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\__pycache__\\test_common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\__pycache__\\test_datetime.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\__pycache__\\test_extension.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\__pycache__\\test_interval.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\__pycache__\\test_masked.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\__pycache__\\test_numpy.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\__pycache__\\test_period.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\__pycache__\\test_sparse.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\__pycache__\\test_string.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\array_with_attr\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\array_with_attr\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\array_with_attr\\__pycache__\\array.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\array_with_attr\\__pycache__\\test_array_with_attr.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\array_with_attr\\array.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\array_with_attr\\test_array_with_attr.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\__pycache__\\accumulate.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\__pycache__\\base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\__pycache__\\casting.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\__pycache__\\constructors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\__pycache__\\dim2.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\__pycache__\\dtype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\__pycache__\\getitem.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\__pycache__\\groupby.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\__pycache__\\index.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\__pycache__\\interface.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\__pycache__\\io.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\__pycache__\\methods.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\__pycache__\\missing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\__pycache__\\ops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\__pycache__\\printing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\__pycache__\\reduce.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\__pycache__\\reshaping.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\__pycache__\\setitem.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\accumulate.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\casting.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\constructors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\dim2.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\dtype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\getitem.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\groupby.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\index.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\interface.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\io.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\methods.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\missing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\ops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\printing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\reduce.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\reshaping.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\base\\setitem.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\date\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\date\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\date\\__pycache__\\array.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\date\\array.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\decimal\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\decimal\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\decimal\\__pycache__\\array.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\decimal\\__pycache__\\test_decimal.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\decimal\\array.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\decimal\\test_decimal.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\json\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\json\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\json\\__pycache__\\array.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\json\\__pycache__\\test_json.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\json\\array.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\json\\test_json.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\list\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\list\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\list\\__pycache__\\array.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\list\\__pycache__\\test_list.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\extension\\list\\array.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\list\\test_list.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\test_arrow.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\test_categorical.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\test_common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\test_datetime.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\test_extension.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\test_interval.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\test_masked.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\test_numpy.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\test_period.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\test_sparse.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\extension\\test_string.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\__pycache__\\common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\__pycache__\\test_alter_axes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\__pycache__\\test_api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\__pycache__\\test_arithmetic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\__pycache__\\test_arrow_interface.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\__pycache__\\test_block_internals.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\__pycache__\\test_constructors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\__pycache__\\test_cumulative.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\__pycache__\\test_iteration.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\__pycache__\\test_logical_ops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\__pycache__\\test_nonunique_indexes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\__pycache__\\test_npfuncs.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\__pycache__\\test_query_eval.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\__pycache__\\test_reductions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\__pycache__\\test_repr.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\__pycache__\\test_stack_unstack.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\__pycache__\\test_subclass.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\__pycache__\\test_ufunc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\__pycache__\\test_unary.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\__pycache__\\test_validate.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\constructors\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\constructors\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\constructors\\__pycache__\\test_from_dict.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\constructors\\__pycache__\\test_from_records.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\constructors\\test_from_dict.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\constructors\\test_from_records.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\__pycache__\\test_coercion.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\__pycache__\\test_delitem.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\__pycache__\\test_get.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\__pycache__\\test_get_value.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\__pycache__\\test_getitem.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\__pycache__\\test_insert.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\__pycache__\\test_mask.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\__pycache__\\test_set_value.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\__pycache__\\test_setitem.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\__pycache__\\test_take.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\__pycache__\\test_where.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\__pycache__\\test_xs.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\test_coercion.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\test_delitem.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\test_get.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\test_get_value.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\test_getitem.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\test_insert.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\test_mask.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\test_set_value.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\test_setitem.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\test_take.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\test_where.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\indexing\\test_xs.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_add_prefix_suffix.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_align.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_asfreq.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_asof.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_assign.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_astype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_at_time.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_between_time.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_clip.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_combine.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_combine_first.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_compare.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_convert_dtypes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_copy.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_count.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_cov_corr.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_describe.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_diff.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_dot.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_drop.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_drop_duplicates.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_droplevel.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_dropna.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_dtypes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_duplicated.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_equals.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_explode.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_fillna.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_filter.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_first_and_last.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_first_valid_index.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_get_numeric_data.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_head_tail.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_infer_objects.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_info.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_interpolate.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_is_homogeneous_dtype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_isetitem.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_isin.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_iterrows.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_join.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_map.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_matmul.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_nlargest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_pct_change.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_pipe.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_pop.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_quantile.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_rank.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_reindex.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_reindex_like.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_rename.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_rename_axis.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_reorder_levels.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_replace.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_reset_index.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_round.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_sample.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_select_dtypes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_set_axis.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_set_index.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_shift.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_size.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_sort_index.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_sort_values.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_swapaxes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_swaplevel.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_to_csv.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_to_dict.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_to_dict_of_blocks.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_to_numpy.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_to_period.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_to_records.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_to_timestamp.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_transpose.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_truncate.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_tz_convert.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_tz_localize.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_update.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_value_counts.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\__pycache__\\test_values.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_add_prefix_suffix.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_align.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_asfreq.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_asof.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_assign.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_astype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_at_time.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_between_time.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_clip.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_combine.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_combine_first.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_compare.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_convert_dtypes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_copy.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_count.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_cov_corr.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_describe.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_diff.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_dot.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_drop.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_drop_duplicates.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_droplevel.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_dropna.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_dtypes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_duplicated.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_equals.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_explode.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_fillna.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_filter.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_first_and_last.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_first_valid_index.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_get_numeric_data.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_head_tail.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_infer_objects.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_info.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_interpolate.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_is_homogeneous_dtype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_isetitem.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_isin.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_iterrows.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_join.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_map.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_matmul.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_nlargest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_pct_change.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_pipe.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_pop.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_quantile.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_rank.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_reindex.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_reindex_like.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_rename.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_rename_axis.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_reorder_levels.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_replace.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_reset_index.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_round.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_sample.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_select_dtypes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_set_axis.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_set_index.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_shift.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_size.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_sort_index.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_sort_values.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_swapaxes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_swaplevel.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_to_csv.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_to_dict.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_to_dict_of_blocks.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_to_numpy.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_to_period.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_to_records.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_to_timestamp.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_transpose.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_truncate.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_tz_convert.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_tz_localize.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_update.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_value_counts.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\methods\\test_values.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\test_alter_axes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\test_api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\test_arithmetic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\test_arrow_interface.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\test_block_internals.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\test_constructors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\test_cumulative.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\test_iteration.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\test_logical_ops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\test_nonunique_indexes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\test_npfuncs.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\test_query_eval.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\test_reductions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\test_repr.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\test_stack_unstack.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\test_subclass.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\test_ufunc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\test_unary.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\frame\\test_validate.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\generic\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\generic\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\generic\\__pycache__\\test_duplicate_labels.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\generic\\__pycache__\\test_finalize.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\generic\\__pycache__\\test_frame.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\generic\\__pycache__\\test_generic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\generic\\__pycache__\\test_label_or_level_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\generic\\__pycache__\\test_series.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\generic\\__pycache__\\test_to_xarray.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\generic\\test_duplicate_labels.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\generic\\test_finalize.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\generic\\test_frame.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\generic\\test_generic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\generic\\test_label_or_level_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\generic\\test_series.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\generic\\test_to_xarray.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_all_methods.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_apply.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_apply_mutate.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_bin_groupby.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_categorical.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_counting.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_cumulative.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_filters.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_groupby.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_groupby_dropna.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_groupby_subclass.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_grouping.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_index_as_string.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_libgroupby.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_missing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_numba.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_numeric_only.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_pipe.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_raises.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_reductions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\__pycache__\\test_timegrouper.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\aggregate\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\aggregate\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\aggregate\\__pycache__\\test_aggregate.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\aggregate\\__pycache__\\test_cython.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\aggregate\\__pycache__\\test_numba.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\aggregate\\__pycache__\\test_other.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\aggregate\\test_aggregate.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\aggregate\\test_cython.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\aggregate\\test_numba.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\aggregate\\test_other.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\__pycache__\\test_corrwith.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\__pycache__\\test_describe.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\__pycache__\\test_groupby_shift_diff.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\__pycache__\\test_is_monotonic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\__pycache__\\test_nlargest_nsmallest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\__pycache__\\test_nth.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\__pycache__\\test_quantile.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\__pycache__\\test_rank.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\__pycache__\\test_sample.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\__pycache__\\test_size.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\__pycache__\\test_skew.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\__pycache__\\test_value_counts.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\test_corrwith.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\test_describe.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\test_groupby_shift_diff.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\test_is_monotonic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\test_nlargest_nsmallest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\test_nth.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\test_quantile.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\test_rank.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\test_sample.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\test_size.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\test_skew.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\methods\\test_value_counts.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_all_methods.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_apply.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_apply_mutate.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_bin_groupby.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_categorical.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_counting.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_cumulative.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_filters.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_groupby.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_groupby_dropna.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_groupby_subclass.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_grouping.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_index_as_string.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_libgroupby.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_missing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_numba.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_numeric_only.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_pipe.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_raises.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_reductions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\test_timegrouper.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\transform\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\transform\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\transform\\__pycache__\\test_numba.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\transform\\__pycache__\\test_transform.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\groupby\\transform\\test_numba.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\groupby\\transform\\test_transform.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\__pycache__\\test_any_index.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\__pycache__\\test_base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\__pycache__\\test_common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\__pycache__\\test_datetimelike.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\__pycache__\\test_engines.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\__pycache__\\test_frozen.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\__pycache__\\test_index_new.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\__pycache__\\test_numpy_compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\__pycache__\\test_old_base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\__pycache__\\test_setops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\__pycache__\\test_subclass.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\base_class\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\base_class\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\base_class\\__pycache__\\test_constructors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\base_class\\__pycache__\\test_formats.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\base_class\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\base_class\\__pycache__\\test_pickle.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\base_class\\__pycache__\\test_reshape.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\base_class\\__pycache__\\test_setops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\base_class\\__pycache__\\test_where.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\base_class\\test_constructors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\base_class\\test_formats.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\base_class\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\base_class\\test_pickle.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\base_class\\test_reshape.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\base_class\\test_setops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\base_class\\test_where.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\__pycache__\\test_append.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\__pycache__\\test_astype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\__pycache__\\test_category.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\__pycache__\\test_constructors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\__pycache__\\test_equals.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\__pycache__\\test_fillna.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\__pycache__\\test_formats.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\__pycache__\\test_map.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\__pycache__\\test_reindex.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\__pycache__\\test_setops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\test_append.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\test_astype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\test_category.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\test_constructors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\test_equals.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\test_fillna.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\test_formats.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\test_map.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\test_reindex.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\categorical\\test_setops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimelike_\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimelike_\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimelike_\\__pycache__\\test_drop_duplicates.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimelike_\\__pycache__\\test_equals.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimelike_\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimelike_\\__pycache__\\test_is_monotonic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimelike_\\__pycache__\\test_nat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimelike_\\__pycache__\\test_sort_values.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimelike_\\__pycache__\\test_value_counts.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimelike_\\test_drop_duplicates.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimelike_\\test_equals.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimelike_\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimelike_\\test_is_monotonic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimelike_\\test_nat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimelike_\\test_sort_values.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimelike_\\test_value_counts.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\__pycache__\\test_arithmetic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\__pycache__\\test_constructors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\__pycache__\\test_date_range.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\__pycache__\\test_datetime.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\__pycache__\\test_formats.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\__pycache__\\test_freq_attr.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\__pycache__\\test_iter.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\__pycache__\\test_join.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\__pycache__\\test_npfuncs.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\__pycache__\\test_ops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\__pycache__\\test_partial_slicing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\__pycache__\\test_pickle.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\__pycache__\\test_reindex.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\__pycache__\\test_scalar_compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\__pycache__\\test_setops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\__pycache__\\test_timezones.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\test_asof.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\test_astype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\test_delete.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\test_factorize.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\test_fillna.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\test_insert.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\test_isocalendar.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\test_map.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\test_normalize.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\test_repeat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\test_resolution.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\test_round.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\test_shift.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\test_snap.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\test_to_frame.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\test_to_julian_date.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\test_to_period.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\test_to_pydatetime.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\test_to_series.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\test_tz_convert.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\test_tz_localize.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\__pycache__\\test_unique.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\test_asof.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\test_astype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\test_delete.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\test_factorize.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\test_fillna.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\test_insert.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\test_isocalendar.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\test_map.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\test_normalize.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\test_repeat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\test_resolution.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\test_round.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\test_shift.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\test_snap.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\test_to_frame.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\test_to_julian_date.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\test_to_period.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\test_to_pydatetime.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\test_to_series.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\test_tz_convert.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\test_tz_localize.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\methods\\test_unique.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\test_arithmetic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\test_constructors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\test_date_range.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\test_datetime.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\test_formats.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\test_freq_attr.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\test_iter.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\test_join.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\test_npfuncs.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\test_ops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\test_partial_slicing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\test_pickle.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\test_reindex.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\test_scalar_compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\test_setops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\datetimes\\test_timezones.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\__pycache__\\test_astype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\__pycache__\\test_constructors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\__pycache__\\test_equals.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\__pycache__\\test_formats.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\__pycache__\\test_interval.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\__pycache__\\test_interval_range.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\__pycache__\\test_interval_tree.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\__pycache__\\test_join.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\__pycache__\\test_pickle.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\__pycache__\\test_setops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\test_astype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\test_constructors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\test_equals.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\test_formats.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\test_interval.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\test_interval_range.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\test_interval_tree.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\test_join.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\test_pickle.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\interval\\test_setops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_analytics.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_astype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_constructors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_conversion.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_copy.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_drop.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_duplicates.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_equivalence.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_formats.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_get_level_values.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_get_set.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_integrity.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_isin.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_join.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_lexsort.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_missing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_monotonic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_names.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_partial_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_pickle.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_reindex.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_reshape.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_setops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_sorting.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\__pycache__\\test_take.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_analytics.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_astype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_constructors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_conversion.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_copy.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_drop.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_duplicates.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_equivalence.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_formats.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_get_level_values.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_get_set.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_integrity.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_isin.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_join.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_lexsort.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_missing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_monotonic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_names.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_partial_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_pickle.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_reindex.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_reshape.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_setops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_sorting.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\multi\\test_take.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\numeric\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\numeric\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\numeric\\__pycache__\\test_astype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\numeric\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\numeric\\__pycache__\\test_join.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\numeric\\__pycache__\\test_numeric.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\numeric\\__pycache__\\test_setops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\numeric\\test_astype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\numeric\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\numeric\\test_join.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\numeric\\test_numeric.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\numeric\\test_setops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\object\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\object\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\object\\__pycache__\\test_astype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\object\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\object\\test_astype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\object\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\__pycache__\\test_constructors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\__pycache__\\test_formats.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\__pycache__\\test_freq_attr.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\__pycache__\\test_join.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\__pycache__\\test_monotonic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\__pycache__\\test_partial_slicing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\__pycache__\\test_period.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\__pycache__\\test_period_range.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\__pycache__\\test_pickle.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\__pycache__\\test_resolution.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\__pycache__\\test_scalar_compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\__pycache__\\test_searchsorted.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\__pycache__\\test_setops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\__pycache__\\test_tools.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\methods\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\methods\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\methods\\__pycache__\\test_asfreq.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\methods\\__pycache__\\test_astype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\methods\\__pycache__\\test_factorize.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\methods\\__pycache__\\test_fillna.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\methods\\__pycache__\\test_insert.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\methods\\__pycache__\\test_is_full.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\methods\\__pycache__\\test_repeat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\methods\\__pycache__\\test_shift.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\methods\\__pycache__\\test_to_timestamp.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\methods\\test_asfreq.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\methods\\test_astype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\methods\\test_factorize.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\methods\\test_fillna.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\methods\\test_insert.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\methods\\test_is_full.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\methods\\test_repeat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\methods\\test_shift.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\methods\\test_to_timestamp.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\test_constructors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\test_formats.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\test_freq_attr.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\test_join.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\test_monotonic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\test_partial_slicing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\test_period.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\test_period_range.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\test_pickle.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\test_resolution.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\test_scalar_compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\test_searchsorted.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\test_setops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\period\\test_tools.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\ranges\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\ranges\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\ranges\\__pycache__\\test_constructors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\ranges\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\ranges\\__pycache__\\test_join.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\ranges\\__pycache__\\test_range.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\ranges\\__pycache__\\test_setops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\ranges\\test_constructors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\ranges\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\ranges\\test_join.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\ranges\\test_range.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\ranges\\test_setops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\test_any_index.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\test_base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\test_common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\test_datetimelike.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\test_engines.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\test_frozen.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\test_index_new.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\test_numpy_compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\test_old_base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\test_setops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\test_subclass.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\__pycache__\\test_arithmetic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\__pycache__\\test_constructors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\__pycache__\\test_delete.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\__pycache__\\test_formats.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\__pycache__\\test_freq_attr.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\__pycache__\\test_join.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\__pycache__\\test_ops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\__pycache__\\test_pickle.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\__pycache__\\test_scalar_compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\__pycache__\\test_searchsorted.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\__pycache__\\test_setops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\__pycache__\\test_timedelta.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\__pycache__\\test_timedelta_range.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\methods\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\methods\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\methods\\__pycache__\\test_astype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\methods\\__pycache__\\test_factorize.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\methods\\__pycache__\\test_fillna.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\methods\\__pycache__\\test_insert.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\methods\\__pycache__\\test_repeat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\methods\\__pycache__\\test_shift.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\methods\\test_astype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\methods\\test_factorize.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\methods\\test_fillna.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\methods\\test_insert.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\methods\\test_repeat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\methods\\test_shift.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\test_arithmetic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\test_constructors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\test_delete.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\test_formats.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\test_freq_attr.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\test_join.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\test_ops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\test_pickle.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\test_scalar_compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\test_searchsorted.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\test_setops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\test_timedelta.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexes\\timedeltas\\test_timedelta_range.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\__pycache__\\common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\__pycache__\\test_at.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\__pycache__\\test_categorical.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\__pycache__\\test_chaining_and_caching.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\__pycache__\\test_check_indexer.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\__pycache__\\test_coercion.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\__pycache__\\test_datetime.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\__pycache__\\test_floats.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\__pycache__\\test_iat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\__pycache__\\test_iloc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\__pycache__\\test_indexers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\__pycache__\\test_loc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\__pycache__\\test_na_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\__pycache__\\test_partial.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\__pycache__\\test_scalar.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\interval\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\interval\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\interval\\__pycache__\\test_interval.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\interval\\__pycache__\\test_interval_new.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\interval\\test_interval.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\interval\\test_interval_new.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\__pycache__\\test_chaining_and_caching.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\__pycache__\\test_datetime.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\__pycache__\\test_getitem.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\__pycache__\\test_iloc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\__pycache__\\test_indexing_slow.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\__pycache__\\test_loc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\__pycache__\\test_multiindex.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\__pycache__\\test_partial.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\__pycache__\\test_setitem.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\__pycache__\\test_slice.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\__pycache__\\test_sorted.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\test_chaining_and_caching.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\test_datetime.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\test_getitem.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\test_iloc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\test_indexing_slow.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\test_loc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\test_multiindex.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\test_partial.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\test_setitem.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\test_slice.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\multiindex\\test_sorted.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\test_at.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\test_categorical.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\test_chaining_and_caching.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\test_check_indexer.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\test_coercion.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\test_datetime.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\test_floats.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\test_iat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\test_iloc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\test_indexers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\test_loc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\test_na_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\test_partial.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\indexing\\test_scalar.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\interchange\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\interchange\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\interchange\\__pycache__\\test_impl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\interchange\\__pycache__\\test_spec_conformance.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\interchange\\__pycache__\\test_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\interchange\\test_impl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\interchange\\test_spec_conformance.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\interchange\\test_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\internals\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\internals\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\internals\\__pycache__\\test_api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\internals\\__pycache__\\test_internals.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\internals\\__pycache__\\test_managers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\internals\\test_api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\internals\\test_internals.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\internals\\test_managers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\__pycache__\\generate_legacy_storage_files.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\__pycache__\\test_clipboard.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\__pycache__\\test_common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\__pycache__\\test_compression.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\__pycache__\\test_feather.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\__pycache__\\test_fsspec.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\__pycache__\\test_gbq.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\__pycache__\\test_gcs.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\__pycache__\\test_html.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\__pycache__\\test_http_headers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\__pycache__\\test_orc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\__pycache__\\test_parquet.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\__pycache__\\test_pickle.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\__pycache__\\test_s3.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\__pycache__\\test_spss.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\__pycache__\\test_sql.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\__pycache__\\test_stata.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\excel\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\excel\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\excel\\__pycache__\\test_odf.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\excel\\__pycache__\\test_odswriter.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\excel\\__pycache__\\test_openpyxl.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\excel\\__pycache__\\test_readers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\excel\\__pycache__\\test_style.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\excel\\__pycache__\\test_writers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\excel\\__pycache__\\test_xlrd.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\excel\\__pycache__\\test_xlsxwriter.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\excel\\test_odf.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\excel\\test_odswriter.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\excel\\test_openpyxl.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\excel\\test_readers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\excel\\test_style.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\excel\\test_writers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\excel\\test_xlrd.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\excel\\test_xlsxwriter.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\__pycache__\\test_console.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\__pycache__\\test_css.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\__pycache__\\test_eng_formatting.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\__pycache__\\test_format.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\__pycache__\\test_ipython_compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\__pycache__\\test_printing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\__pycache__\\test_to_csv.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\__pycache__\\test_to_excel.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\__pycache__\\test_to_html.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\__pycache__\\test_to_latex.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\__pycache__\\test_to_markdown.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\__pycache__\\test_to_string.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\__pycache__\\test_bar.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\__pycache__\\test_exceptions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\__pycache__\\test_format.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\__pycache__\\test_highlight.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\__pycache__\\test_html.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\__pycache__\\test_matplotlib.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\__pycache__\\test_non_unique.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\__pycache__\\test_style.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\__pycache__\\test_to_latex.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\__pycache__\\test_to_string.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\__pycache__\\test_tooltip.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\test_bar.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\test_exceptions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\test_format.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\test_highlight.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\test_html.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\test_matplotlib.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\test_non_unique.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\test_style.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\test_to_latex.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\test_to_string.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\style\\test_tooltip.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\test_console.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\test_css.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\test_eng_formatting.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\test_format.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\test_ipython_compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\test_printing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\test_to_csv.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\test_to_excel.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\test_to_html.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\test_to_latex.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\test_to_markdown.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\formats\\test_to_string.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\generate_legacy_storage_files.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\json\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\json\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\json\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\json\\__pycache__\\test_compression.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\json\\__pycache__\\test_deprecated_kwargs.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\json\\__pycache__\\test_json_table_schema.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\json\\__pycache__\\test_json_table_schema_ext_dtype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\json\\__pycache__\\test_normalize.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\json\\__pycache__\\test_pandas.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\json\\__pycache__\\test_readlines.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\json\\__pycache__\\test_ujson.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\json\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\json\\test_compression.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\json\\test_deprecated_kwargs.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\json\\test_json_table_schema.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\json\\test_json_table_schema_ext_dtype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\json\\test_normalize.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\json\\test_pandas.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\json\\test_readlines.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\json\\test_ujson.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\test_c_parser_only.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\test_comment.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\test_compression.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\test_concatenate_chunks.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\test_converters.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\test_dialect.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\test_encoding.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\test_header.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\test_index_col.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\test_mangle_dupes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\test_multi_thread.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\test_na_values.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\test_network.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\test_parse_dates.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\test_python_parser_only.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\test_quoting.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\test_read_fwf.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\test_skiprows.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\test_textreader.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\test_unsupported.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\__pycache__\\test_upcast.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\__pycache__\\test_chunksize.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\__pycache__\\test_common_basic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\__pycache__\\test_data_list.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\__pycache__\\test_decimal.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\__pycache__\\test_file_buffer_url.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\__pycache__\\test_float.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\__pycache__\\test_index.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\__pycache__\\test_inf.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\__pycache__\\test_ints.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\__pycache__\\test_iterator.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\__pycache__\\test_read_errors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\__pycache__\\test_verbose.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\test_chunksize.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\test_common_basic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\test_data_list.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\test_decimal.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\test_file_buffer_url.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\test_float.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\test_index.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\test_inf.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\test_ints.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\test_iterator.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\test_read_errors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\common\\test_verbose.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\dtypes\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\dtypes\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\dtypes\\__pycache__\\test_categorical.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\dtypes\\__pycache__\\test_dtypes_basic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\dtypes\\__pycache__\\test_empty.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\dtypes\\test_categorical.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\dtypes\\test_dtypes_basic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\dtypes\\test_empty.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\test_c_parser_only.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\test_comment.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\test_compression.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\test_concatenate_chunks.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\test_converters.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\test_dialect.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\test_encoding.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\test_header.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\test_index_col.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\test_mangle_dupes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\test_multi_thread.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\test_na_values.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\test_network.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\test_parse_dates.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\test_python_parser_only.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\test_quoting.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\test_read_fwf.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\test_skiprows.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\test_textreader.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\test_unsupported.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\test_upcast.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\usecols\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\usecols\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\usecols\\__pycache__\\test_parse_dates.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\usecols\\__pycache__\\test_strings.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\usecols\\__pycache__\\test_usecols_basic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\usecols\\test_parse_dates.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\usecols\\test_strings.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\parser\\usecols\\test_usecols_basic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\__pycache__\\common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\__pycache__\\test_append.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\__pycache__\\test_categorical.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\__pycache__\\test_compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\__pycache__\\test_complex.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\__pycache__\\test_errors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\__pycache__\\test_file_handling.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\__pycache__\\test_keys.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\__pycache__\\test_put.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\__pycache__\\test_pytables_missing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\__pycache__\\test_read.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\__pycache__\\test_retain_attributes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\__pycache__\\test_round_trip.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\__pycache__\\test_select.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\__pycache__\\test_store.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\__pycache__\\test_subclass.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\__pycache__\\test_time_series.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\__pycache__\\test_timezones.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\test_append.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\test_categorical.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\test_compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\test_complex.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\test_errors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\test_file_handling.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\test_keys.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\test_put.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\test_pytables_missing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\test_read.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\test_retain_attributes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\test_round_trip.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\test_select.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\test_store.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\test_subclass.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\test_time_series.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\pytables\\test_timezones.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\sas\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\sas\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\sas\\__pycache__\\test_byteswap.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\sas\\__pycache__\\test_sas.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\sas\\__pycache__\\test_sas7bdat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\sas\\__pycache__\\test_xport.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\sas\\test_byteswap.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\sas\\test_sas.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\sas\\test_sas7bdat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\sas\\test_xport.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\test_clipboard.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\test_common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\test_compression.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\test_feather.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\test_fsspec.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\test_gbq.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\test_gcs.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\test_html.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\test_http_headers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\test_orc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\test_parquet.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\test_pickle.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\test_s3.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\test_spss.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\test_sql.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\test_stata.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\xml\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\xml\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\xml\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\xml\\__pycache__\\test_to_xml.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\xml\\__pycache__\\test_xml.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\xml\\__pycache__\\test_xml_dtypes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\io\\xml\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\xml\\test_to_xml.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\xml\\test_xml.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\io\\xml\\test_xml_dtypes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\libs\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\libs\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\libs\\__pycache__\\test_hashtable.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\libs\\__pycache__\\test_join.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\libs\\__pycache__\\test_lib.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\libs\\__pycache__\\test_libalgos.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\libs\\test_hashtable.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\libs\\test_join.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\libs\\test_lib.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\libs\\test_libalgos.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\plotting\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\plotting\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\plotting\\__pycache__\\common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\plotting\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\plotting\\__pycache__\\test_backend.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\plotting\\__pycache__\\test_boxplot_method.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\plotting\\__pycache__\\test_common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\plotting\\__pycache__\\test_converter.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\plotting\\__pycache__\\test_datetimelike.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\plotting\\__pycache__\\test_groupby.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\plotting\\__pycache__\\test_hist_method.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\plotting\\__pycache__\\test_misc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\plotting\\__pycache__\\test_series.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\plotting\\__pycache__\\test_style.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\plotting\\common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\plotting\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\plotting\\frame\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\plotting\\frame\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\plotting\\frame\\__pycache__\\test_frame.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\plotting\\frame\\__pycache__\\test_frame_color.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\plotting\\frame\\__pycache__\\test_frame_groupby.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\plotting\\frame\\__pycache__\\test_frame_legend.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\plotting\\frame\\__pycache__\\test_frame_subplots.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\plotting\\frame\\__pycache__\\test_hist_box_by.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\plotting\\frame\\test_frame.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\plotting\\frame\\test_frame_color.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\plotting\\frame\\test_frame_groupby.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\plotting\\frame\\test_frame_legend.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\plotting\\frame\\test_frame_subplots.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\plotting\\frame\\test_hist_box_by.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\plotting\\test_backend.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\plotting\\test_boxplot_method.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\plotting\\test_common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\plotting\\test_converter.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\plotting\\test_datetimelike.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\plotting\\test_groupby.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\plotting\\test_hist_method.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\plotting\\test_misc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\plotting\\test_series.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\plotting\\test_style.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reductions\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reductions\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reductions\\__pycache__\\test_reductions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reductions\\__pycache__\\test_stat_reductions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reductions\\test_reductions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reductions\\test_stat_reductions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\resample\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\resample\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\resample\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\resample\\__pycache__\\test_base.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\resample\\__pycache__\\test_datetime_index.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\resample\\__pycache__\\test_period_index.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\resample\\__pycache__\\test_resample_api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\resample\\__pycache__\\test_resampler_grouper.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\resample\\__pycache__\\test_time_grouper.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\resample\\__pycache__\\test_timedelta.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\resample\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\resample\\test_base.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\resample\\test_datetime_index.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\resample\\test_period_index.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\resample\\test_resample_api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\resample\\test_resampler_grouper.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\resample\\test_time_grouper.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\resample\\test_timedelta.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\__pycache__\\test_crosstab.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\__pycache__\\test_cut.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\__pycache__\\test_from_dummies.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\__pycache__\\test_get_dummies.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\__pycache__\\test_melt.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\__pycache__\\test_pivot.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\__pycache__\\test_pivot_multilevel.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\__pycache__\\test_qcut.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\__pycache__\\test_union_categoricals.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\__pycache__\\test_util.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\__pycache__\\test_append.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\__pycache__\\test_append_common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\__pycache__\\test_categorical.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\__pycache__\\test_concat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\__pycache__\\test_dataframe.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\__pycache__\\test_datetimes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\__pycache__\\test_empty.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\__pycache__\\test_index.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\__pycache__\\test_invalid.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\__pycache__\\test_series.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\__pycache__\\test_sort.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\test_append.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\test_append_common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\test_categorical.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\test_concat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\test_dataframe.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\test_datetimes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\test_empty.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\test_index.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\test_invalid.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\test_series.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\concat\\test_sort.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\merge\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\merge\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\merge\\__pycache__\\test_join.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\merge\\__pycache__\\test_merge.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\merge\\__pycache__\\test_merge_asof.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\merge\\__pycache__\\test_merge_cross.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\merge\\__pycache__\\test_merge_index_as_string.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\merge\\__pycache__\\test_merge_ordered.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\merge\\__pycache__\\test_multi.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\reshape\\merge\\test_join.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\merge\\test_merge.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\merge\\test_merge_asof.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\merge\\test_merge_cross.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\merge\\test_merge_index_as_string.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\merge\\test_merge_ordered.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\merge\\test_multi.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\test_crosstab.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\test_cut.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\test_from_dummies.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\test_get_dummies.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\test_melt.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\test_pivot.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\test_pivot_multilevel.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\test_qcut.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\test_union_categoricals.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\reshape\\test_util.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\__pycache__\\test_na_scalar.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\__pycache__\\test_nat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\interval\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\interval\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\interval\\__pycache__\\test_arithmetic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\interval\\__pycache__\\test_constructors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\interval\\__pycache__\\test_contains.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\interval\\__pycache__\\test_formats.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\interval\\__pycache__\\test_interval.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\interval\\__pycache__\\test_overlaps.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\interval\\test_arithmetic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\interval\\test_constructors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\interval\\test_contains.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\interval\\test_formats.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\interval\\test_interval.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\interval\\test_overlaps.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\period\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\period\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\period\\__pycache__\\test_arithmetic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\period\\__pycache__\\test_asfreq.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\period\\__pycache__\\test_period.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\period\\test_arithmetic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\period\\test_asfreq.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\period\\test_period.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\test_na_scalar.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\test_nat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timedelta\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timedelta\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timedelta\\__pycache__\\test_arithmetic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timedelta\\__pycache__\\test_constructors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timedelta\\__pycache__\\test_formats.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timedelta\\__pycache__\\test_timedelta.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timedelta\\methods\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timedelta\\methods\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timedelta\\methods\\__pycache__\\test_as_unit.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timedelta\\methods\\__pycache__\\test_round.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timedelta\\methods\\test_as_unit.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timedelta\\methods\\test_round.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timedelta\\test_arithmetic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timedelta\\test_constructors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timedelta\\test_formats.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timedelta\\test_timedelta.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\__pycache__\\test_arithmetic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\__pycache__\\test_comparisons.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\__pycache__\\test_constructors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\__pycache__\\test_formats.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\__pycache__\\test_timestamp.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\__pycache__\\test_timezones.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\methods\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\methods\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\methods\\__pycache__\\test_as_unit.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\methods\\__pycache__\\test_normalize.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\methods\\__pycache__\\test_replace.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\methods\\__pycache__\\test_round.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\methods\\__pycache__\\test_timestamp_method.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\methods\\__pycache__\\test_to_julian_date.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\methods\\__pycache__\\test_to_pydatetime.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\methods\\__pycache__\\test_tz_convert.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\methods\\__pycache__\\test_tz_localize.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\methods\\test_as_unit.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\methods\\test_normalize.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\methods\\test_replace.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\methods\\test_round.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\methods\\test_timestamp_method.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\methods\\test_to_julian_date.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\methods\\test_to_pydatetime.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\methods\\test_tz_convert.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\methods\\test_tz_localize.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\test_arithmetic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\test_comparisons.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\test_constructors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\test_formats.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\test_timestamp.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\scalar\\timestamp\\test_timezones.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\__pycache__\\test_api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\__pycache__\\test_arithmetic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\__pycache__\\test_constructors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\__pycache__\\test_cumulative.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\__pycache__\\test_formats.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\__pycache__\\test_iteration.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\__pycache__\\test_logical_ops.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\__pycache__\\test_missing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\__pycache__\\test_npfuncs.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\__pycache__\\test_reductions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\__pycache__\\test_subclass.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\__pycache__\\test_ufunc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\__pycache__\\test_unary.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\__pycache__\\test_validate.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\accessors\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\accessors\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\accessors\\__pycache__\\test_cat_accessor.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\accessors\\__pycache__\\test_dt_accessor.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\accessors\\__pycache__\\test_list_accessor.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\accessors\\__pycache__\\test_sparse_accessor.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\accessors\\__pycache__\\test_str_accessor.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\accessors\\__pycache__\\test_struct_accessor.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\accessors\\test_cat_accessor.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\accessors\\test_dt_accessor.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\accessors\\test_list_accessor.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\accessors\\test_sparse_accessor.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\accessors\\test_str_accessor.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\accessors\\test_struct_accessor.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\__pycache__\\test_datetime.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\__pycache__\\test_delitem.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\__pycache__\\test_get.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\__pycache__\\test_getitem.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\__pycache__\\test_indexing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\__pycache__\\test_mask.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\__pycache__\\test_set_value.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\__pycache__\\test_setitem.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\__pycache__\\test_take.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\__pycache__\\test_where.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\__pycache__\\test_xs.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\test_datetime.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\test_delitem.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\test_get.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\test_getitem.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\test_indexing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\test_mask.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\test_set_value.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\test_setitem.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\test_take.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\test_where.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\indexing\\test_xs.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_add_prefix_suffix.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_align.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_argsort.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_asof.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_astype.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_autocorr.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_between.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_case_when.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_clip.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_combine.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_combine_first.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_compare.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_convert_dtypes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_copy.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_count.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_cov_corr.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_describe.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_diff.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_drop.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_drop_duplicates.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_dropna.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_dtypes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_duplicated.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_equals.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_explode.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_fillna.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_get_numeric_data.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_head_tail.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_infer_objects.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_info.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_interpolate.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_is_monotonic.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_is_unique.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_isin.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_isna.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_item.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_map.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_matmul.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_nlargest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_nunique.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_pct_change.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_pop.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_quantile.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_rank.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_reindex.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_reindex_like.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_rename.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_rename_axis.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_repeat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_replace.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_reset_index.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_round.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_searchsorted.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_set_name.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_size.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_sort_index.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_sort_values.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_to_csv.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_to_dict.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_to_frame.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_to_numpy.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_tolist.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_truncate.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_tz_localize.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_unique.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_unstack.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_update.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_value_counts.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_values.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\__pycache__\\test_view.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_add_prefix_suffix.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_align.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_argsort.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_asof.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_astype.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_autocorr.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_between.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_case_when.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_clip.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_combine.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_combine_first.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_compare.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_convert_dtypes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_copy.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_count.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_cov_corr.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_describe.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_diff.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_drop.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_drop_duplicates.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_dropna.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_dtypes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_duplicated.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_equals.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_explode.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_fillna.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_get_numeric_data.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_head_tail.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_infer_objects.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_info.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_interpolate.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_is_monotonic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_is_unique.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_isin.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_isna.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_item.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_map.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_matmul.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_nlargest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_nunique.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_pct_change.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_pop.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_quantile.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_rank.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_reindex.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_reindex_like.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_rename.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_rename_axis.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_repeat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_replace.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_reset_index.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_round.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_searchsorted.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_set_name.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_size.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_sort_index.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_sort_values.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_to_csv.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_to_dict.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_to_frame.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_to_numpy.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_tolist.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_truncate.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_tz_localize.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_unique.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_unstack.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_update.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_value_counts.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_values.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\methods\\test_view.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\test_api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\test_arithmetic.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\test_constructors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\test_cumulative.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\test_formats.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\test_iteration.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\test_logical_ops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\test_missing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\test_npfuncs.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\test_reductions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\test_subclass.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\test_ufunc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\test_unary.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\series\\test_validate.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\strings\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\strings\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\strings\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\strings\\__pycache__\\test_api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\strings\\__pycache__\\test_case_justify.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\strings\\__pycache__\\test_cat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\strings\\__pycache__\\test_extract.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\strings\\__pycache__\\test_find_replace.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\strings\\__pycache__\\test_get_dummies.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\strings\\__pycache__\\test_split_partition.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\strings\\__pycache__\\test_string_array.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\strings\\__pycache__\\test_strings.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\strings\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\strings\\test_api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\strings\\test_case_justify.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\strings\\test_cat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\strings\\test_extract.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\strings\\test_find_replace.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\strings\\test_get_dummies.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\strings\\test_split_partition.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\strings\\test_string_array.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\strings\\test_strings.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\test_aggregation.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\test_algos.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\test_common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\test_downstream.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\test_errors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\test_expressions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\test_flags.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\test_multilevel.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\test_nanops.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\test_optional_dependency.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\test_register_accessor.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\test_sorting.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\test_take.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tools\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tools\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tools\\__pycache__\\test_to_datetime.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tools\\__pycache__\\test_to_numeric.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tools\\__pycache__\\test_to_time.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tools\\__pycache__\\test_to_timedelta.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tools\\test_to_datetime.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tools\\test_to_numeric.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tools\\test_to_time.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tools\\test_to_timedelta.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\frequencies\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\frequencies\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\frequencies\\__pycache__\\test_freq_code.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\frequencies\\__pycache__\\test_frequencies.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\frequencies\\__pycache__\\test_inference.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\frequencies\\test_freq_code.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\frequencies\\test_frequencies.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\frequencies\\test_inference.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\holiday\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\holiday\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\holiday\\__pycache__\\test_calendar.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\holiday\\__pycache__\\test_federal.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\holiday\\__pycache__\\test_holiday.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\holiday\\__pycache__\\test_observance.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\holiday\\test_calendar.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\holiday\\test_federal.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\holiday\\test_holiday.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\holiday\\test_observance.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__pycache__\\common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__pycache__\\test_business_day.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__pycache__\\test_business_hour.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__pycache__\\test_business_month.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__pycache__\\test_business_quarter.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__pycache__\\test_business_year.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__pycache__\\test_common.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__pycache__\\test_custom_business_day.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__pycache__\\test_custom_business_hour.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__pycache__\\test_custom_business_month.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__pycache__\\test_dst.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__pycache__\\test_easter.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__pycache__\\test_fiscal.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__pycache__\\test_index.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__pycache__\\test_month.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__pycache__\\test_offsets.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__pycache__\\test_offsets_properties.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__pycache__\\test_quarter.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__pycache__\\test_ticks.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__pycache__\\test_week.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\__pycache__\\test_year.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\test_business_day.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\test_business_hour.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\test_business_month.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\test_business_quarter.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\test_business_year.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\test_common.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\test_custom_business_day.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\test_custom_business_hour.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\test_custom_business_month.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\test_dst.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\test_easter.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\test_fiscal.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\test_index.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\test_month.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\test_offsets.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\test_offsets_properties.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\test_quarter.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\test_ticks.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\test_week.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tseries\\offsets\\test_year.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\__pycache__\\test_api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\__pycache__\\test_array_to_datetime.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\__pycache__\\test_ccalendar.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\__pycache__\\test_conversion.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\__pycache__\\test_fields.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\__pycache__\\test_libfrequencies.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\__pycache__\\test_liboffsets.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\__pycache__\\test_np_datetime.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\__pycache__\\test_npy_units.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\__pycache__\\test_parse_iso8601.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\__pycache__\\test_parsing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\__pycache__\\test_period.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\__pycache__\\test_resolution.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\__pycache__\\test_strptime.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\__pycache__\\test_timedeltas.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\__pycache__\\test_timezones.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\__pycache__\\test_to_offset.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\__pycache__\\test_tzconversion.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\test_api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\test_array_to_datetime.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\test_ccalendar.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\test_conversion.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\test_fields.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\test_libfrequencies.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\test_liboffsets.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\test_np_datetime.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\test_npy_units.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\test_parse_iso8601.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\test_parsing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\test_period.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\test_resolution.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\test_strptime.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\test_timedeltas.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\test_timezones.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\test_to_offset.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\tslibs\\test_tzconversion.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_assert_almost_equal.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_assert_attr_equal.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_assert_categorical_equal.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_assert_extension_array_equal.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_assert_frame_equal.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_assert_index_equal.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_assert_interval_array_equal.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_assert_numpy_array_equal.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_assert_produces_warning.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_assert_series_equal.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_deprecate.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_deprecate_kwarg.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_deprecate_nonkeyword_arguments.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_doc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_hashing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_numba.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_rewrite_warning.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_shares_memory.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_show_versions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_util.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_validate_args.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_validate_args_and_kwargs.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_validate_inclusive.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\__pycache__\\test_validate_kwargs.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\util\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_assert_almost_equal.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_assert_attr_equal.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_assert_categorical_equal.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_assert_extension_array_equal.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_assert_frame_equal.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_assert_index_equal.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_assert_interval_array_equal.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_assert_numpy_array_equal.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_assert_produces_warning.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_assert_series_equal.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_deprecate.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_deprecate_kwarg.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_deprecate_nonkeyword_arguments.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_doc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_hashing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_numba.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_rewrite_warning.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_shares_memory.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_show_versions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_util.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_validate_args.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_validate_args_and_kwargs.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_validate_inclusive.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\util\\test_validate_kwargs.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\__pycache__\\test_api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\__pycache__\\test_apply.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\__pycache__\\test_base_indexer.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\__pycache__\\test_cython_aggregations.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\__pycache__\\test_dtypes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\__pycache__\\test_ewm.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\__pycache__\\test_expanding.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\__pycache__\\test_groupby.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\__pycache__\\test_numba.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\__pycache__\\test_online.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\__pycache__\\test_pairwise.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\__pycache__\\test_rolling.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\__pycache__\\test_rolling_functions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\__pycache__\\test_rolling_quantile.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\__pycache__\\test_rolling_skew_kurt.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\__pycache__\\test_timeseries_window.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\__pycache__\\test_win_type.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\moments\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\moments\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\moments\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\moments\\__pycache__\\test_moments_consistency_ewm.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\moments\\__pycache__\\test_moments_consistency_expanding.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\moments\\__pycache__\\test_moments_consistency_rolling.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tests\\window\\moments\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\moments\\test_moments_consistency_ewm.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\moments\\test_moments_consistency_expanding.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\moments\\test_moments_consistency_rolling.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\test_api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\test_apply.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\test_base_indexer.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\test_cython_aggregations.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\test_dtypes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\test_ewm.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\test_expanding.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\test_groupby.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\test_numba.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\test_online.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\test_pairwise.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\test_rolling.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\test_rolling_functions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\test_rolling_quantile.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\test_rolling_skew_kurt.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\test_timeseries_window.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tests\\window\\test_win_type.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tseries\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tseries\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tseries\\__pycache__\\api.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tseries\\__pycache__\\frequencies.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tseries\\__pycache__\\holiday.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tseries\\__pycache__\\offsets.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\tseries\\api.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tseries\\frequencies.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tseries\\holiday.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\tseries\\offsets.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\util\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\util\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\util\\__pycache__\\_decorators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\util\\__pycache__\\_doctools.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\util\\__pycache__\\_exceptions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\util\\__pycache__\\_print_versions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\util\\__pycache__\\_test_decorators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\util\\__pycache__\\_tester.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\util\\__pycache__\\_validators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pandas\\util\\_decorators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\util\\_doctools.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\util\\_exceptions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\util\\_print_versions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\util\\_test_decorators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\util\\_tester.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\util\\_validators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\util\\version\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pandas\\util\\version\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\past\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\past\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\past\\builtins\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\past\\builtins\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\past\\builtins\\__pycache__\\misc.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\past\\builtins\\__pycache__\\noniterators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\past\\builtins\\misc.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\past\\builtins\\noniterators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\past\\translation\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\past\\translation\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\past\\types\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\past\\types\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\past\\types\\__pycache__\\basestring.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\past\\types\\__pycache__\\olddict.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\past\\types\\__pycache__\\oldstr.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\past\\types\\basestring.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\past\\types\\olddict.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\past\\types\\oldstr.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\past\\utils\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\past\\utils\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika-1.3.2.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika-1.3.2.dist-info\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika-1.3.2.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika-1.3.2.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika-1.3.2.dist-info\\REQUESTED', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika-1.3.2.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika-1.3.2.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika-1.3.2.dist-info\\zip-safe', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\__pycache__\\amqp_object.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\__pycache__\\callback.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\__pycache__\\channel.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\__pycache__\\compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\__pycache__\\connection.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\__pycache__\\credentials.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\__pycache__\\data.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\__pycache__\\delivery_mode.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\__pycache__\\diagnostic_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\__pycache__\\exceptions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\__pycache__\\exchange_type.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\__pycache__\\frame.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\__pycache__\\heartbeat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\__pycache__\\spec.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\__pycache__\\tcp_socket_opts.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\__pycache__\\validators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\adapters\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\adapters\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\adapters\\__pycache__\\asyncio_connection.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\adapters\\__pycache__\\base_connection.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\adapters\\__pycache__\\blocking_connection.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\adapters\\__pycache__\\gevent_connection.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\adapters\\__pycache__\\select_connection.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\adapters\\__pycache__\\tornado_connection.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\adapters\\__pycache__\\twisted_connection.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\adapters\\asyncio_connection.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\adapters\\base_connection.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\adapters\\blocking_connection.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\adapters\\gevent_connection.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\adapters\\select_connection.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\adapters\\tornado_connection.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\adapters\\twisted_connection.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\adapters\\utils\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\adapters\\utils\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\adapters\\utils\\__pycache__\\connection_workflow.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\adapters\\utils\\__pycache__\\io_services_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\adapters\\utils\\__pycache__\\nbio_interface.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\adapters\\utils\\__pycache__\\selector_ioloop_adapter.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pika\\adapters\\utils\\connection_workflow.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\adapters\\utils\\io_services_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\adapters\\utils\\nbio_interface.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\adapters\\utils\\selector_ioloop_adapter.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\amqp_object.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\callback.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\channel.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\connection.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\credentials.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\data.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\delivery_mode.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\diagnostic_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\exceptions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\exchange_type.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\frame.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\heartbeat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\spec.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\tcp_socket_opts.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pika\\validators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\psycopg2\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\psycopg2\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\psycopg2\\__pycache__\\_ipaddress.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\psycopg2\\__pycache__\\_json.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\psycopg2\\__pycache__\\_range.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\psycopg2\\__pycache__\\errorcodes.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\psycopg2\\__pycache__\\errors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\psycopg2\\__pycache__\\extensions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\psycopg2\\__pycache__\\extras.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\psycopg2\\__pycache__\\pool.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\psycopg2\\__pycache__\\sql.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\psycopg2\\__pycache__\\tz.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\psycopg2\\_ipaddress.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\psycopg2\\_json.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\psycopg2\\_psycopg.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\psycopg2\\_range.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\psycopg2\\errorcodes.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\psycopg2\\errors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\psycopg2\\extensions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\psycopg2\\extras.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\psycopg2\\pool.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\psycopg2\\sql.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\psycopg2\\tz.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\psycopg2_binary-2.9.10.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\psycopg2_binary-2.9.10.dist-info\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\psycopg2_binary-2.9.10.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\psycopg2_binary-2.9.10.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\psycopg2_binary-2.9.10.dist-info\\REQUESTED', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\psycopg2_binary-2.9.10.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\psycopg2_binary-2.9.10.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic-2.10.6.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic-2.10.6.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic-2.10.6.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic-2.10.6.dist-info\\REQUESTED', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic-2.10.6.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic-2.10.6.dist-info\\licenses\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\_migration.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\alias_generators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\aliases.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\annotated_handlers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\class_validators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\color.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\config.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\dataclasses.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\datetime_parse.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\decorator.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\env_settings.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\error_wrappers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\errors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\fields.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\functional_serializers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\functional_validators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\generics.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\json.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\json_schema.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\main.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\mypy.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\networks.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\parse.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\root_model.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\schema.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\tools.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\type_adapter.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\types.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\typing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\validate_call_decorator.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\validators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\__pycache__\\warnings.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_config.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_core_metadata.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_core_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_dataclasses.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_decorators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_decorators_v1.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_discriminated_union.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_docs_extraction.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_fields.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_forward_ref.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_generate_schema.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_generics.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_git.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_import_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_internal_dataclass.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_known_annotated_metadata.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_mock_val_ser.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_model_construction.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_namespace_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_repr.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_schema_generation_shared.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_serializers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_signature.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_std_types_schema.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_typing_extra.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_validate_call.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\__pycache__\\_validators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\_internal\\_config.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_core_metadata.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_core_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_dataclasses.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_decorators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_decorators_v1.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_discriminated_union.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_docs_extraction.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_fields.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_forward_ref.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_generate_schema.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_generics.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_git.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_import_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_internal_dataclass.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_known_annotated_metadata.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_mock_val_ser.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_model_construction.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_namespace_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_repr.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_schema_generation_shared.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_serializers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_signature.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_std_types_schema.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_typing_extra.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_validate_call.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_internal\\_validators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\_migration.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\alias_generators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\aliases.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\annotated_handlers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\class_validators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\color.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\config.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\dataclasses.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\datetime_parse.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\decorator.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\deprecated\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\deprecated\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\deprecated\\__pycache__\\class_validators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\deprecated\\__pycache__\\config.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\deprecated\\__pycache__\\copy_internals.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\deprecated\\__pycache__\\decorator.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\deprecated\\__pycache__\\json.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\deprecated\\__pycache__\\parse.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\deprecated\\__pycache__\\tools.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\deprecated\\class_validators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\deprecated\\config.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\deprecated\\copy_internals.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\deprecated\\decorator.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\deprecated\\json.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\deprecated\\parse.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\deprecated\\tools.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\env_settings.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\error_wrappers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\errors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\experimental\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\experimental\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\experimental\\__pycache__\\pipeline.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\experimental\\pipeline.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\fields.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\functional_serializers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\functional_validators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\generics.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\json.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\json_schema.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\main.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\mypy.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\networks.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\parse.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\plugin\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\plugin\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\plugin\\__pycache__\\_loader.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\plugin\\__pycache__\\_schema_validator.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\plugin\\_loader.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\plugin\\_schema_validator.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\root_model.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\schema.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\tools.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\type_adapter.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\types.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\typing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\_hypothesis_plugin.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\annotated_types.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\class_validators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\color.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\config.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\dataclasses.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\datetime_parse.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\decorator.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\env_settings.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\error_wrappers.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\errors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\fields.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\generics.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\json.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\main.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\mypy.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\networks.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\parse.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\schema.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\tools.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\types.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\typing.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\validators.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\__pycache__\\version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic\\v1\\_hypothesis_plugin.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\annotated_types.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\class_validators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\color.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\config.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\dataclasses.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\datetime_parse.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\decorator.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\env_settings.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\error_wrappers.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\errors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\fields.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\generics.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\json.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\main.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\mypy.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\networks.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\parse.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\schema.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\tools.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\types.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\typing.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\validators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\v1\\version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\validate_call_decorator.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\validators.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic\\warnings.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic_core-2.27.2.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic_core-2.27.2.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic_core-2.27.2.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic_core-2.27.2.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic_core-2.27.2.dist-info\\licenses\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic_core\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic_core\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic_core\\__pycache__\\core_schema.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic_core\\_pydantic_core.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pydantic_core\\_pydantic_core.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic_core\\core_schema.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pydantic_core\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyodbc-5.2.0.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyodbc-5.2.0.dist-info\\LICENSE.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyodbc-5.2.0.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyodbc-5.2.0.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyodbc-5.2.0.dist-info\\REQUESTED', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyodbc-5.2.0.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyodbc-5.2.0.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyodbc.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyodbc.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio-0.10.0.dist-info\\DELVEWHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio-0.10.0.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio-0.10.0.dist-info\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio-0.10.0.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio-0.10.0.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio-0.10.0.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio-0.10.0.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio.libs\\Lerc-089e3fef3df84b17326dcddbf1dedaa4.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio.libs\\gdal-debee5933f0da7bb90b4bcd009023377.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio.libs\\geos-ace4c5b5c1f569bb4213e7bbd0b0322e.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio.libs\\geos_c-7478ca0a86136b280d9b2d245c6f6627.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio.libs\\geotiff-c8fe8a095520a4ea4e465d27e06add3a.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio.libs\\iconv-2-27352d156a5467ca5383d3951093ea5a.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio.libs\\jpeg62-e56b6f95a95af498f4623b8da4cebd46.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio.libs\\json-c-c84940e2654a4f8468bfcf2ce992aa93.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio.libs\\libcurl-d69cfd4ad487d53d58743b6778ec85e7.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio.libs\\libexpat-6576a8d02641b6a3dbad35901ec200a7.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio.libs\\liblzma-9ee4accb476ec1ae24e924953140273d.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio.libs\\libpng16-7c36142dda59f186f6bb683e8dae2bfe.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio.libs\\msvcp140-98b3e5b80de1e5e9d1703b786d795623.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio.libs\\proj-a408c5327f3fd2f5fabe8c56815beed7.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio.libs\\qhull_r-516897f855568caab1ab1fe37912766c.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio.libs\\sqlite3-9bc109d8536d5ed9666332fec94485fc.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio.libs\\tiff-9b3f605fffe0bccc0a964c374ee4f820.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio.libs\\zlib1-e5af16a15c63f05bd82d90396807ae5b.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\__pycache__\\_compat.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\__pycache__\\_env.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\__pycache__\\_version.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\__pycache__\\core.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\__pycache__\\errors.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\__pycache__\\geopandas.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\__pycache__\\raw.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\__pycache__\\util.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\_compat.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\_env.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\_err.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\_err.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\_geometry.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\_geometry.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\_io.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\_io.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\_ogr.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\_ogr.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\_version.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\_vsi.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\_vsi.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\core.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\errors.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\GDAL-targets-release.cmake', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\GDAL-targets.cmake', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\GDALConfig.cmake', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\GDALConfigVersion.cmake', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\GDALLogoBW.svg', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\GDALLogoColor.svg', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\GDALLogoGS.svg', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\LICENSE.TXT', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\MM_m_idofic.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\gdal_data\\copyright', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\cubewerx_extra.wkt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\default.rsc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\gdal_data\\ecw_cs.wkt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\eedaconf.json', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\epsg.wkt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\esri_StatePlane_extra.wkt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\gdalicon.png', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\gdal_data\\gdalinfo_output.schema.json', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\gdalmdiminfo_output.schema.json', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\gdaltileindex.xsd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\gdalvrt.xsd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\gfs.xsd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\gml_registry.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\gml_registry.xsd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_center.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_process.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_subcenter.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_0_0.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_0_1.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_0_13.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_0_14.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_0_15.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_0_16.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_0_17.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_0_18.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_0_19.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_0_190.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_0_191.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_0_2.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_0_20.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_0_21.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_0_3.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_0_4.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_0_5.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_0_6.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_0_7.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_10_0.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_10_1.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_10_191.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_10_2.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_10_3.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_10_4.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_1_0.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_1_1.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_1_2.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_20_0.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_20_1.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_20_2.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_2_0.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_2_3.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_2_4.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_2_5.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_2_6.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_3_0.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_3_1.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_3_2.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_3_3.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_3_4.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_3_5.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_3_6.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_4_0.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_4_1.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_4_10.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_4_2.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_4_3.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_4_4.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_4_5.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_4_6.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_4_7.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_4_8.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_4_9.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_local_Canada.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_local_HPC.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_local_MRMS.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_local_NCEP.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_local_NDFD.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_2_local_index.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_4_5.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\grib2_table_versions.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\gt_datum.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\gt_ellips.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\header.dxf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\inspire_cp_BasicPropertyUnit.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\inspire_cp_CadastralBoundary.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\inspire_cp_CadastralParcel.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\inspire_cp_CadastralZoning.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_AdmArea.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_AdmBdry.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_AdmPt.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_BldA.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_BldL.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_Cntr.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_CommBdry.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_CommPt.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_Cstline.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_ElevPt.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_GCP.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_LeveeEdge.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_RailCL.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_RdASL.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_RdArea.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_RdCompt.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_RdEdg.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_RdMgtBdry.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_RdSgmtA.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_RvrMgtBdry.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_SBAPt.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_SBArea.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_SBBdry.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_WA.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_WL.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_WStrA.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\jpfgdgml_WStrL.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\nitf_spec.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\nitf_spec.xsd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\ogrinfo_output.schema.json', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\ogrvrt.xsd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\osmconf.ini', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\ozi_datum.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\ozi_ellips.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\pci_datum.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\pci_ellips.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\pdfcomposition.xsd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\pds4_template.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\plscenesconf.json', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\ruian_vf_ob_v1.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\ruian_vf_st_uvoh_v1.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\ruian_vf_st_v1.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\ruian_vf_v1.gfs', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\s57agencies.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\s57attributes.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\s57expectedinput.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\gdal_data\\s57objectclasses.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\seed_2d.dgn', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\gdal_data\\seed_3d.dgn', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\gdal_data\\stateplane.csv', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\tms_LINZAntarticaMapTileGrid.json', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\tms_MapML_APSTILE.json', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\tms_MapML_CBMTILE.json', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\tms_NZTM2000.json', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\trailer.dxf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\usage', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\vcpkg-cmake-wrapper.cmake', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\vcpkg.spdx.json', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\vcpkg_abi_info.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\vdv452.xml', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\vdv452.xsd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\gdal_data\\vicar.json', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\geopandas.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\CH', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\GL27', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\ITRF2000', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\ITRF2008', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\ITRF2014', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\copyright', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\deformation_model.schema.json', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\nad.lst', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\nad27', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\nad83', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\other.extra', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\proj-config-version.cmake', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\proj-config.cmake', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\proj-targets-release.cmake', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\proj-targets.cmake', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\proj.db', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\proj_data\\proj.ini', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\proj4-targets-release.cmake', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\proj4-targets.cmake', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\projjson.schema.json', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\triangulation.schema.json', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\usage', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\vcpkg.spdx.json', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\vcpkg_abi_info.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\proj_data\\world', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\raw.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\tests\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\tests\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\tests\\__pycache__\\conftest.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\tests\\__pycache__\\test_arrow.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\tests\\__pycache__\\test_core.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\tests\\__pycache__\\test_geopandas_io.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\tests\\__pycache__\\test_path.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\tests\\__pycache__\\test_raw_io.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\tests\\__pycache__\\test_util.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\tests\\conftest.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\tests\\fixtures\\README.md', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\tests\\fixtures\\curve.gpkg', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\tests\\fixtures\\curvepolygon.gpkg', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\tests\\fixtures\\line_zm.gpkg', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\tests\\fixtures\\multisurface.gpkg', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\tests\\fixtures\\naturalearth_lowres\\naturalearth_lowres.cpg', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\tests\\fixtures\\naturalearth_lowres\\naturalearth_lowres.dbf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\tests\\fixtures\\naturalearth_lowres\\naturalearth_lowres.prj', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\tests\\fixtures\\naturalearth_lowres\\naturalearth_lowres.shp', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\tests\\fixtures\\naturalearth_lowres\\naturalearth_lowres.shx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\tests\\fixtures\\sample.osm.pbf', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\tests\\fixtures\\test_gpkg_nulls.gpkg', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyogrio\\tests\\test_arrow.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\tests\\test_core.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\tests\\test_geopandas_io.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\tests\\test_path.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\tests\\test_raw_io.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\tests\\test_util.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyogrio\\util.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj-3.7.1.dist-info\\DELVEWHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj-3.7.1.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj-3.7.1.dist-info\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj-3.7.1.dist-info\\LICENSE_proj', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj-3.7.1.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj-3.7.1.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj-3.7.1.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj-3.7.1.dist-info\\entry_points.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj-3.7.1.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj.libs\\jpeg62-b8ab92f4fbc92367f3262d8a1a71f90b.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj.libs\\libcurl-a29cc9317505e6997addcef4c84fa626.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj.libs\\liblzma-559e6637e59731119da5f77161d49851.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj.libs\\msvcp140-d76d4b45e040cbc263297f5a5893a46c.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj.libs\\proj_9-ece80b3ce99919cf7dd31637c074a1bb.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj.libs\\sqlite3-4d959957f8a1ac1287d0dfb03e0e5353.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj.libs\\tiff-5e90040fe2ccc57bb6a51e477dfe595e.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj.libs\\zlib1-20140a261acd3e37526970333bcd3057.dll', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\__main__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\__pycache__\\__main__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\__pycache__\\_show_versions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\__pycache__\\aoi.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\__pycache__\\datadir.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\__pycache__\\enums.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\__pycache__\\exceptions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\__pycache__\\geod.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\__pycache__\\network.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\__pycache__\\proj.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\__pycache__\\sync.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\__pycache__\\transformer.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\__pycache__\\utils.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\_compat.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_compat.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\_compat.pxd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_compat.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_compat.pyx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_context.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_context.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\_context.pxd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_context.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_context.pyx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_crs.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_crs.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\_crs.pxd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_crs.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_crs.pyx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_geod.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_geod.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\_geod.pxd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_geod.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_geod.pyx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_network.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_network.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\_network.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_network.pyx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_show_versions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_sync.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_sync.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\_sync.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_sync.pyx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_transformer.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_transformer.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\_transformer.pxd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_transformer.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_transformer.pyx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_version.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_version.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\_version.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\_version.pyx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\aoi.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\base.pxi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\crs\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\crs\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\crs\\__pycache__\\_cf1x8.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\crs\\__pycache__\\coordinate_operation.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\crs\\__pycache__\\coordinate_system.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\crs\\__pycache__\\crs.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\crs\\__pycache__\\datum.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\crs\\__pycache__\\enums.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\crs\\_cf1x8.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\crs\\coordinate_operation.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\crs\\coordinate_system.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\crs\\crs.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\crs\\datum.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\crs\\enums.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\database.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\database.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\database.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\database.pyx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\datadir.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\enums.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\exceptions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\geod.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\list.c', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\list.cp310-win_amd64.pyd', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\list.pyi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\list.pyx', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\network.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj.pxi', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\CH', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\GL27', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\ITRF2000', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\ITRF2008', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\ITRF2014', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\ITRF2020', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\copyright', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\deformation_model.schema.json', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\nad.lst', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\nad27', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\nad83', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\other.extra', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\proj-config-version.cmake', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\proj-config.cmake', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\proj-targets-debug.cmake', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\proj-targets-release.cmake', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\proj-targets.cmake', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\proj.db', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\proj.ini', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\proj4-targets-debug.cmake', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\proj4-targets-release.cmake', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\proj4-targets.cmake', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\projjson.schema.json', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\triangulation.schema.json', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\usage', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\vcpkg.spdx.json', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\vcpkg_abi_info.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\proj_dir\\share\\proj\\world', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\py.typed', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\sync.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\transformer.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pyproj\\utils.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\python_dateutil-2.9.0.post0.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\python_dateutil-2.9.0.post0.dist-info\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\python_dateutil-2.9.0.post0.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\python_dateutil-2.9.0.post0.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\python_dateutil-2.9.0.post0.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\python_dateutil-2.9.0.post0.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\python_dateutil-2.9.0.post0.dist-info\\zip-safe', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\python_dotenv-1.0.1.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\python_dotenv-1.0.1.dist-info\\LICENSE', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\python_dotenv-1.0.1.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\python_dotenv-1.0.1.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\python_dotenv-1.0.1.dist-info\\REQUESTED', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\python_dotenv-1.0.1.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\python_dotenv-1.0.1.dist-info\\entry_points.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\python_dotenv-1.0.1.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pytz-2025.1.dist-info\\INSTALLER', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pytz-2025.1.dist-info\\LICENSE.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pytz-2025.1.dist-info\\METADATA', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pytz-2025.1.dist-info\\RECORD', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pytz-2025.1.dist-info\\WHEEL', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pytz-2025.1.dist-info\\top_level.txt', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pytz-2025.1.dist-info\\zip-safe', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pytz\\__init__.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pytz\\__pycache__\\__init__.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\__pycache__\\exceptions.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\__pycache__\\lazy.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\__pycache__\\reference.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\__pycache__\\tzfile.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\__pycache__\\tzinfo.cpython-310.pyc', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\exceptions.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pytz\\lazy.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pytz\\reference.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pytz\\tzfile.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pytz\\tzinfo.py', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'text'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Abidjan', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Accra', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Addis_Ababa', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Algiers', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Asmara', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Asmera', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Bamako', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Bangui', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Banjul', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Bissau', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Blantyre', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Brazzaville', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Bujumbura', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Cairo', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Casablanca', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Ceuta', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Conakry', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Dakar', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Dar_es_Salaam', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Djibouti', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Douala', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\El_Aaiun', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Freetown', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Gaborone', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Harare', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Johannesburg', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Juba', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Kampala', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Khartoum', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Kigali', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Kinshasa', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Lagos', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Libreville', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\\envs\\data_sync', 'binary'),
('Lib\\site-packages\\pytz\\zoneinfo\\Africa\\Lome', 'C:\\Users\\detection.WIN-4ESSIE0VG8T\\miniconda3\